<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once 'koneksi.php';

// Cek login
checkLogin();

// Pastikan zona waktu PHP diatur dengan benar
date_default_timezone_set('Asia/Jakarta');

$user_id = $_SESSION['user_id'];
$role = $_SESSION['role'];
$nik_karyawan = $_SESSION['nik_karyawan'];

// Pastikan hanya peserta atau ISS yang bisa submit
if (!in_array($role, ['peserta', 'iss'])) {
    header("Location: dashboard.php?error=" . urlencode("Anda tidak memiliki izin untuk mengakses halaman ini."));
    exit();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $training_id = $_POST['training_id'] ?? null;
    $submitted_answers = $_POST['answers'] ?? []; // Array: [question_id => selected_option]

    if (!$training_id || empty($submitted_answers)) {
        header("Location: take_posttest.php?training_id=" . htmlspecialchars($training_id) . "&error=" . urlencode("Data jawaban tidak lengkap."));
        exit();
    }

    // Pastikan koneksi database valid
    if (!$koneksi) {
        error_log("Database connection failed in submit_posttest.php");
        echo "<!DOCTYPE html><html lang='en'><head><meta charset='UTF-8'><meta name='viewport' content='width=device-width, initial-scale=1.0'><title>Error Koneksi Database</title><link rel='stylesheet' href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css'></head><body><div class='container mt-5'><div class='alert alert-danger'><h3>Kesalahan Koneksi Database!</h3><p>Sistem tidak dapat terhubung ke database. Mohon hubungi administrator.</p></div></div></body></html>";
        exit();
    }

    $koneksi->begin_transaction();
    $correct_answers_count = 0;
    $total_questions_answered = 0;

    try {
        // Generate timestamp in PHP (which is already set to Asia/Jakarta timezone)
        $current_submission_time = date('Y-m-d H:i:s');

        // 1. Insert a new record into post_test_submissions
        // Explicitly include submission_time with the PHP-generated timestamp
        $stmt = $koneksi->prepare("INSERT INTO post_test_submissions (user_id, user_role, training_id, score, submission_time) VALUES (?, ?, ?, NULL, ?)");
        if (!$stmt) {
            throw new Exception("Gagal mempersiapkan statement untuk submission: " . $koneksi->error);
        }
        // 'isis' -> integer, string, integer, string (for user_id, role, training_id, current_submission_time)
        $stmt->bind_param("isis", $user_id, $role, $training_id, $current_submission_time);
        if (!$stmt->execute()) {
            throw new Exception("Gagal menyimpan data submission: " . $stmt->error);
        }
        $submission_id = $stmt->insert_id;
        $stmt->close();

        // 2. Get correct answers for the training from the database
        $question_ids_in_submission = array_keys($submitted_answers);
        if (empty($question_ids_in_submission)) {
            throw new Exception("Tidak ada soal yang dijawab dalam submission.");
        }

        // Buat placeholder untuk IN clause
        $placeholders = implode(',', array_fill(0, count($question_ids_in_submission), '?'));
        $types = str_repeat('i', count($question_ids_in_submission)); // Tipe data untuk question IDs

        $query_get_correct_answers = "SELECT id, correct_answer FROM post_test_questions WHERE id IN ($placeholders) AND training_id = ?";
        $stmt = $koneksi->prepare($query_get_correct_answers);
        if (!$stmt) {
            throw new Exception("Gagal mempersiapkan statement untuk mengambil jawaban benar: " . $koneksi->error . " Query: " . $query_get_correct_answers);
        }

        // Gabungkan array ID soal dan training_id untuk binding
        $bind_values = array_merge($question_ids_in_submission, [$training_id]);
        $types .= 'i'; // Tambahkan tipe data untuk training_id

        // Buat array argumen untuk bind_param, dimulai dengan string tipe data
        $params_to_bind = [$types];
        foreach ($bind_values as $key => $value) {
            $params_to_bind[] = &$bind_values[$key]; // Pastikan setiap nilai adalah referensi
        }

        // Bind parameter menggunakan splat operator (membutuhkan PHP 5.6+)
        $stmt->bind_param(...$params_to_bind);

        if (!$stmt->execute()) {
            throw new Exception("Gagal mengambil jawaban benar dari database: " . $stmt->error);
        }
        $result = $stmt->get_result();
        $correct_answers_map = [];
        while ($row = $result->fetch_assoc()) {
            // Trim dan ubah ke uppercase saat mengambil dari DB untuk perbandingan yang konsisten
            $correct_answers_map[$row['id']] = strtoupper(trim($row['correct_answer']));
        }
        $stmt->close();

        // 3. Process each submitted answer and insert into post_test_answers
        $stmt_insert_answer = $koneksi->prepare("INSERT INTO post_test_answers (submission_id, question_id, selected_option, is_correct) VALUES (?, ?, ?, ?)");
        if (!$stmt_insert_answer) {
            throw new Exception("Gagal mempersiapkan statement untuk menyimpan jawaban: " . $koneksi->error);
        }

        foreach ($submitted_answers as $question_id => $selected_option) {
            $total_questions_answered++; // Hitung total soal yang dijawab
            $is_correct = 0; // Default ke salah

            // Trim dan ubah ke uppercase untuk jawaban yang disubmit
            $trimmed_selected_option = strtoupper(trim($selected_option));

            if (isset($correct_answers_map[$question_id])) {
                $correct_ans_from_db = $correct_answers_map[$question_id]; // Sudah di-trim dan uppercase saat diambil

                if ($correct_ans_from_db === $trimmed_selected_option) {
                    $is_correct = 1; // Benar
                    $correct_answers_count++; // Hitung jumlah jawaban benar
                }
            }

            // Simpan selected_option asli (tanpa trim/uppercase) ke database jika diinginkan,
            // atau simpan yang sudah di-trim/uppercase jika ingin konsisten.
            $stmt_insert_answer->bind_param("iisi", $submission_id, $question_id, $selected_option, $is_correct);
            if (!$stmt_insert_answer->execute()) {
                throw new Exception("Gagal menyimpan jawaban soal ID " . $question_id . ": " . $stmt_insert_answer->error);
            }
        }
        $stmt_insert_answer->close();

        // 4. Calculate final score (1-100 scale) and update the post_test_submissions table
        $final_score_100 = 0;
        if ($total_questions_answered > 0) {
            $final_score_100 = round(($correct_answers_count / $total_questions_answered) * 100);
        }

        $stmt = $koneksi->prepare("UPDATE post_test_submissions SET score = ? WHERE id = ?");
        if (!$stmt) {
            throw new Exception("Gagal mempersiapkan statement untuk update skor: " . $koneksi->error);
        }
        $stmt->bind_param("ii", $final_score_100, $submission_id); // Gunakan $final_score_100
        if (!$stmt->execute()) {
            throw new Exception("Gagal memperbarui skor submission: " . $stmt->error);
        }
        $stmt->close();

        $koneksi->commit(); // Commit transaksi jika semua berhasil
        header("Location: post_test_result.php?submission_id=" . $submission_id);
        exit();

    } catch (Exception $e) {
        $koneksi->rollback(); // Rollback jika ada error
        error_log("Error submitting post test: " . $e->getMessage()); // Log error ke server log
        echo "<!DOCTYPE html><html lang='en'><head><meta charset='UTF-8'><meta name='viewport' content='width=device-width, initial-scale=1.0'><title>Error</title><link rel='stylesheet' href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css'></head><body><div class='container mt-5'><div class='alert alert-danger'><h3>Terjadi Kesalahan!</h3><p>Mohon maaf, terjadi kesalahan saat menyimpan jawaban Anda. Silakan coba lagi.</p><p>Detail Error (untuk debugging): " . htmlspecialchars($e->getMessage()) . "</p><a href='take_posttest.php?training_id=" . htmlspecialchars($training_id) . "' class='btn btn-primary'>Kembali ke Post Test</a></div></div></body></html>";
        exit();
    }
} else {
    // Redirect jika bukan request POST
    header("Location: dashboard.php");
    exit();
}
?>
