<?php
require_once 'koneksi.php';

// Cek jika user sudah login
if (isset($_SESSION['user_id'])) {
    // Redirect ke dashboard untuk semua role (tidak ada pembatasan khusus ISS)
    redirect("dashboard.php");
}

$error = "";

// Proses login
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $nik_karyawan = clean($_POST['nik_karyawan']);
    $password = $_POST['password'] ?? '';
    $role = clean($_POST['role']);
    
    // Debug: Log role yang dipilih
    error_log("Login attempt with role: " . $role);
    
    // Validasi role yang diizinkan
    $allowed_roles = ['trainer', 'peserta', 'iss'];
    if (!in_array($role, $allowed_roles)) {
        $error = "Role yang dipilih tidak valid";
        error_log("Invalid role selected: " . $role);
    }
    // Validasi berdasarkan role
    elseif (empty($nik_karyawan)) {
        $error = "NIK Karyawan harus diisi";
    } elseif ($role != 'iss' && empty($password)) {
        $error = "Password harus diisi untuk login sebagai " . ucfirst($role);
    } elseif (!preg_match('/^[a-zA-Z0-9]+$/', $nik_karyawan)) {
        $error = "NIK Karyawan hanya boleh berisi huruf dan angka";
    } else {
        // Tentukan tabel berdasarkan role
        $table = '';
        switch ($role) {
            case 'trainer':
                $table = 'trainers';
                break;
            case 'peserta':
                $table = 'peserta';
                break;
            case 'iss':
                $table = 'iss';
                break;
            default:
                $error = "Role tidak dikenali";
                break;
        }
        
        if (empty($error) && !empty($table)) {
            try {
                // Untuk ISS, tidak ada kolom password, jadi hanya cek NIK
                if ($role == 'iss') {
                    error_log("Attempting ISS login for NIK: " . $nik_karyawan);
                    
                    $stmt = $koneksi->prepare("SELECT id, nik_karyawan FROM $table WHERE nik_karyawan = ?");
                    $stmt->bind_param("s", $nik_karyawan);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    
                    if ($result->num_rows == 1) {
                        $row = $result->fetch_assoc();
                        
                        error_log("ISS user found, ID: " . $row['id']);
                        
                        // Update last_login untuk ISS
                        $update_stmt = $koneksi->prepare("UPDATE $table SET last_login = NOW() WHERE id = ?");
                        $update_stmt->bind_param("i", $row['id']);
                        $update_stmt->execute();
                        $update_stmt->close();
                        
                        // Set session - ISS diperlakukan sama seperti peserta
                        $_SESSION['user_id'] = $row['id'];
                        $_SESSION['nik_karyawan'] = $row['nik_karyawan'];
                        $_SESSION['role'] = 'iss';
                        
                        error_log("ISS login successful, redirecting to dashboard.php");
                        
                        // ISS sekarang diarahkan ke dashboard.php seperti peserta
                        redirect("dashboard.php");
                    } else {
                        $error = "NIK Karyawan tidak ditemukan di database ISS";
                        error_log("ISS NIK not found: " . $nik_karyawan);
                    }
                    $stmt->close();
                } else {
                    // Untuk trainer dan peserta, cek password
                    error_log("Attempting " . $role . " login for NIK: " . $nik_karyawan);
                    
                    $stmt = $koneksi->prepare("SELECT id, nik_karyawan, password FROM $table WHERE nik_karyawan = ?");
                    $stmt->bind_param("s", $nik_karyawan);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    
                    if ($result->num_rows == 1) {
                        $row = $result->fetch_assoc();
                        if (password_verify($password, $row['password'])) {
                            $_SESSION['user_id'] = $row['id'];
                            $_SESSION['nik_karyawan'] = $row['nik_karyawan'];
                            $_SESSION['role'] = $role;
                            
                            error_log($role . " login successful, redirecting to dashboard.php");
                            redirect("dashboard.php");
                        } else {
                            $error = "Password tidak valid";
                            error_log("Invalid password for " . $role . " NIK: " . $nik_karyawan);
                        }
                    } else {
                        $error = "NIK Karyawan tidak ditemukan";
                        error_log($role . " NIK not found: " . $nik_karyawan);
                    }
                    $stmt->close();
                }
            } catch (Exception $e) {
                $error = "Terjadi kesalahan sistem. Silakan coba lagi.";
                error_log("Login error: " . $e->getMessage());
            }
        }
    }
}

// Jika ada error dari URL parameter (redirect dari tempat lain)
if (isset($_GET['error'])) {
    $error = urldecode($_GET['error']);
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PIM Training</title>
    <link rel="icon" type="image/png" href="image/icon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        html, body {
            height: 100%;
            margin: 0;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background-image: url('image/bg.jpg'); /* Updated background image */
            background-size: cover;
            background-position: center center;
            background-repeat: no-repeat;
            display: flex; /* Mengatur body sebagai flex container */
            flex-direction: column; /* Mengatur item dalam kolom */
            min-height: 100vh; /* Memastikan body mengambil tinggi penuh viewport */
            align-items: stretch; /* Memungkinkan wrapper untuk meregang secara horizontal */
            justify-content: flex-start; /* Memulai konten dari atas, footer akan terdorong ke bawah */
            color: #fff; /* Default text color for elements directly on body */
        }
        
        .wrapper {
            flex-grow: 1; /* Memastikan wrapper mengambil semua ruang yang tersedia */
            display: flex;
            flex-direction: column;
            align-items: center; /* Memusatkan login card secara horizontal */
            justify-content: center; /* Memusatkan login card secara vertikal di dalam ruang wrapper */
            width: 100%; /* Memastikan wrapper mengambil lebar penuh */
            /* min-height: 100vh; dihapus karena body sudah mengatur tinggi keseluruhan */
        }
        
        .login-container {
            max-width: 450px;
            width: 100%;
            padding: 0 15px;
        }
        
        .login-card {
            background: rgba(255, 255, 255, 0.5); /* Semi-transparent white */
            border-radius: 15px;
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.2); /* Stronger shadow */
            overflow: hidden;
            transition: transform 0.3s ease;
            backdrop-filter: blur(5px); /* Add blur for transparency effect */
            color: #333; /* Default text color inside card */
        }
        
        .login-card:hover {
            transform: translateY(-5px);
        }
        
        .login-body {
            padding: 2rem;
        }
        
        .form-label {
            font-weight: 500;
            color: #555; /* Darker grey for readability on light card */
        }
        
        .input-group-text {
            background: #e9ecef; /* Lighter background */
            border: 1px solid #ced4da; /* Subtle border */
            color: #555;
            border-right: none; /* Remove right border to blend with input */
            border-radius: 8px 0 0 8px; /* Match input border-radius */
        }
        
        .form-control, .form-select {
            border: 1px solid #ced4da; /* Subtle border */
            border-radius: 8px;
            padding: 0.75rem;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
            background-color: rgba(255, 255, 255, 0.5); /* Slightly transparent input */
            color: #333; /* Ensure input text is dark */
        }
        
        .form-control:focus, .form-select:focus {
            border-color: #86b9c6; /* A light blue/teal for focus */
            box-shadow: 0 0 0 0.2rem rgba(134, 185, 198, 0.25);
        }
        
        .btn-primary {
            background: linear-gradient(to right, #ff9a9e 0%, #fad0c4 99%, #fad0c4 100%); /* Gradient from target design */
            border: none;
            color: #fff; /* White text */
            padding: 0.75rem;
            font-weight: 500;
            border-radius: 8px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .btn-primary:hover {
            background: linear-gradient(to right, #ff9a9e 0%, #fad0c4 99%, #fad0c4 100%); /* Keep same gradient on hover */
            opacity: 0.9; /* Slight opacity change on hover */
            transform: scale(1.02); /* Slightly smaller scale on hover */
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
        }
        
        .alert {
            border-radius: 8px;
            padding: 1rem;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .auth-footer {
            text-align: center;
            padding: 0.3rem 1rem 1rem 1rem; /* Mengurangi padding atas menjadi 0.5rem */
            background: transparent; /* Make it transparent */
            border-top: none; /* Remove border */
            color: #555; /* Darker text for readability */
        }
        
        .auth-footer a {
            color: #0a939e; /* Keep the existing theme color for the link */
            text-decoration: none;
            font-weight: 500;
        }
        
        .auth-footer a:hover {
            text-decoration: underline;
        }
        
        .toggle-password {
            background: #e9ecef; /* Match input group text background */
            border: 1px solid #ced4da; /* Match input group text border */
            border-left: none; /* Remove left border */
            border-radius: 0 8px 8px 0; /* Match input border-radius */
            color: #555;
        }
        
        .animate-fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        footer {
            flex-shrink: 0; /* Mencegah footer menyusut */
            text-align: center;
            padding: 1rem 0;
            background: transparent; /* Make it transparent to show background image */
            color: #ffffff; /* White text for readability on background image */
            border-top: none; /* Remove border */
            /* position: absolute; bottom: 0; width: 100%; dihapus */
        }

        .password-field {
            display: block;
            transition: all 0.3s ease;
        }
        
        .password-field.hidden {
            display: none;
        }
        
        .debug-info {
            background: #f8f9fa;
            border: 1px solid rgba(157, 206, 218, 0.2);
            border-radius: 8px;
            padding: 0.75rem;
            margin-bottom: 1rem;
            font-size: 0.8rem;
            color: #666666;
        }
        
        @media (max-width: 576px) {
            .login-container {
                padding: 0 10px;
            }
            .login-card {
                border-radius: 10px;
            }
            .login-body {
                padding: 1.5rem;
            }
            .btn-primary {
                padding: 0.65rem;
            }
        }
    </style>
</head>
<body>
    <div class="wrapper">
        <div class="login-container">
            <div class="login-card animate-fade-in">
                <div class="login-body">
                    <h2 class="text-center mb-2" style="color: #555; font-weight: 600;">PIM Training</h2>
                    <p class="text-center mb-4" style="color: #555;">Pondok Indah Mall</p>
                    
                    <?php if(!empty($error)): ?>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle"></i>
                            <?php echo htmlspecialchars($error); ?>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Debug info (hapus setelah testing) -->
                    <?php if (isset($_GET['debug'])): ?>
                        <div class="debug-info">
                            <strong>Debug Info:</strong><br>
                            Session Role: <?php echo $_SESSION['role'] ?? 'Not set'; ?><br>
                            User ID: <?php echo $_SESSION['user_id'] ?? 'Not set'; ?><br>
                            NIK: <?php echo $_SESSION['nik_karyawan'] ?? 'Not set'; ?>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST" action="">
                        <div class="mb-3">
                            <label for="role" class="form-label">Login Sebagai</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-user-tag"></i></span>
                                <select class="form-select" name="role" id="role" required>
                                    <option value="peserta">Peserta</option>
                                    <option value="trainer">Trainer</option>
                                    <option value="iss">ISS</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="nik_karyawan" class="form-label">NIK Karyawan</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-id-badge"></i></span>
                                <input type="text" class="form-control" id="nik_karyawan" name="nik_karyawan" placeholder="NIK Karyawan" required pattern="[a-zA-Z0-9]+" title="NIK Karyawan hanya boleh berisi huruf dan angka" value="<?php echo isset($_POST['nik_karyawan']) ? htmlspecialchars($_POST['nik_karyawan']) : ''; ?>">
                            </div>
                        </div>
                        <div class="mb-4 password-field" id="passwordField">
                            <label for="password" class="form-label">Password</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                <input type="password" class="form-control" id="password" name="password" placeholder="Password">
                                <button class="btn toggle-password" type="button" id="togglePassword">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                        <div class="d-grid mb-4">
                            <button type="submit" class="btn btn-primary">
                                SIGN IN
                            </button>
                        </div>
                        <div class="d-flex justify-content-between align-items-center mb-4">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="" id="rememberMe">
                                <label class="form-check-label" for="rememberMe" style="color: #555;">
                                    Remember Me
                                </label>
                            </div>
                            <a href="#" class="text-decoration-none" style="color: #0a939e;">Forgot Password?</a>
                        </div>
                    </form>
                </div>
                <div class="auth-footer">
                    <p>Registrasi sebagai peserta ISS? <a href="registrasi_iss.php">Daftar di sini</a></p>
                </div>
            </div>
        </div>
        <footer>
            <div class="container">
                <p class="mb-0" style="color: black;">© <?php echo date('Y'); ?> E-Learning System by PIM</p>>
            </div>
        </footer>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle password visibility
        document.getElementById('togglePassword').addEventListener('click', function() {
            const passwordInput = document.getElementById('password');
            const icon = this.querySelector('i');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                passwordInput.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        });

        // Handle role change - hide/show password field
        document.getElementById('role').addEventListener('change', function() {
            const passwordField = document.getElementById('passwordField');
            const passwordInput = document.getElementById('password');
            
            if (this.value === 'iss') {
                // Hide password field for ISS
                passwordField.classList.add('hidden');
                passwordInput.removeAttribute('required');
                passwordInput.value = ''; // Clear password field
            } else {
                // Show password field for Trainer and Peserta
                passwordField.classList.remove('hidden');
                passwordInput.setAttribute('required', 'required');
            }
        });

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            const roleSelect = document.getElementById('role');
            const passwordField = document.getElementById('passwordField');
            const passwordInput = document.getElementById('password');
            
            if (roleSelect.value === 'iss') {
                passwordField.classList.add('hidden');
                passwordInput.removeAttribute('required');
            } else {
                passwordField.classList.remove('hidden');
                passwordInput.setAttribute('required', 'required');
            }
        });
    </script>
</body>
</html>
