<?php
require_once 'koneksi.php';

// Set timezone ke Asia/Jakarta
date_default_timezone_set('Asia/Jakarta');

// TAMBAHAN: Set timezone MySQL agar sama dengan PHP
$koneksi->query("SET time_zone = '+07:00'");

// Cek login
checkLogin();

$user_id = $_SESSION['user_id'];
$role = $_SESSION['role'];
$nik_karyawan = $_SESSION['nik_karyawan'];
$error = "";
$success = "";

// Ambil informasi user menggunakan prepared statement
if ($role == 'trainer') {
    $stmt = $koneksi->prepare("SELECT nama_lengkap, foto_profil, jabatan, divisi FROM trainers WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
} elseif ($role == 'iss') {
    // Untuk ISS, ambil semua kolom yang tersedia
    $stmt = $koneksi->prepare("SELECT nama_lengkap, divisi, jabatan FROM iss WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $iss_data = $result->fetch_assoc();
    $stmt->close();
    
    // Set data user dengan foto_profil null karena tidak ada di tabel iss
    $user = [
        'nama_lengkap' => $iss_data['nama_lengkap'] ?? 'Unknown',
        'foto_profil' => null, // ISS tidak memiliki foto profil
        'jabatan' => $iss_data['jabatan'] ?? 'Tidak ada jabatan',
        'divisi' => $iss_data['divisi'] ?? 'Tidak ada divisi'
    ];
} else {
    $stmt = $koneksi->prepare("SELECT nama_lengkap, foto_profil, jabatan, divisi FROM peserta WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
}

$nama_user = $user['nama_lengkap'] ?? 'Unknown';

// ✅ Proses join training (untuk peserta dan ISS)
if (($role == 'peserta' || $role == 'iss') && isset($_GET['join_id'])) {
    $join_id = (int)$_GET['join_id'];
    
    // ✅ Logika berbeda untuk peserta vs ISS
    if ($role == 'peserta') {
        // Peserta: cek undangan
        $stmt = $koneksi->prepare("
            SELECT t.id, t.judul, t.nama_trainer, t.training_date, t.start_time, t.file_name, t.video_name 
            FROM trainings t 
            LEFT JOIN training_invitations ti ON t.id = ti.training_id 
            WHERE t.id = ? AND (ti.peserta_id = ? OR ti.peserta_id IS NULL) 
            GROUP BY t.id 
            ORDER BY t.training_date DESC, t.start_time DESC
        ");
        $stmt->bind_param("ii", $join_id, $user_id);
    } else if ($role == 'iss') {
        // ISS: langsung bisa join semua training
        $stmt = $koneksi->prepare("
            SELECT id, judul, nama_trainer, training_date, start_time, file_name, video_name 
            FROM trainings 
            WHERE id = ?
        ");
        $stmt->bind_param("i", $join_id);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows == 0) {
        $error = "Training tidak ditemukan atau Anda tidak diundang.";
    } else {
        $training = $result->fetch_assoc();
        $training_datetime = new DateTime($training['training_date'] . ' ' . $training['start_time']);
        // Tambahkan 1 jam ke waktu mulai
        $training_datetime->add(new DateInterval('PT1H'));
        $current_datetime = new DateTime();
        
        // Debug: Log waktu untuk verifikasi
        error_log("Join Attempt: Current Time = " . $current_datetime->format('Y-m-d H:i:s') . 
                 ", Training Time + 1 Hour = " . $training_datetime->format('Y-m-d H:i:s'));
        
        // Perbandingan waktu
        if ($current_datetime >= $training_datetime) {
            $error = "Waktu training telah habis.";
        } else {
            // ✅ Cek apakah user sudah bergabung (support user_type)
            $stmt = $koneksi->prepare("SELECT id FROM training_participants WHERE training_id = ? AND user_id = ? AND user_type = ?");
            $stmt->bind_param("iis", $join_id, $user_id, $role);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows == 0) {
                // ✅ Insert dengan user_type
                $current_time = date('Y-m-d H:i:s');
                $stmt = $koneksi->prepare("INSERT INTO training_participants (training_id, user_id, user_type, join_time) VALUES (?, ?, ?, ?)");
                $stmt->bind_param("iiss", $join_id, $user_id, $role, $current_time);
                if ($stmt->execute()) {
                    $success = "Berhasil bergabung dengan training.";
                } else {
                    $error = "Gagal bergabung dengan training: " . $koneksi->error;
                }
            } else {
                $error = "Anda sudah bergabung dengan training ini.";
            }
        }
    }
    $stmt->close();
}

// Ambil daftar training menggunakan prepared statement
if ($role == 'peserta') {
    // Hanya tampilkan training yang mengundang peserta ini atau yang tidak memiliki undangan khusus
    $stmt = $koneksi->prepare("
        SELECT t.id, t.judul, t.nama_trainer, t.training_date, t.start_time, t.file_name, t.video_name 
        FROM trainings t 
        LEFT JOIN training_invitations ti ON t.id = ti.training_id 
        WHERE ti.peserta_id = ? OR ti.peserta_id IS NULL 
        GROUP BY t.id 
        ORDER BY t.training_date DESC
    ");
    $stmt->bind_param("i", $user_id);
} else if ($role == 'iss') {
    // ✅ ISS bisa akses semua training tanpa perlu undangan
    $stmt = $koneksi->prepare("
        SELECT id, judul, nama_trainer, training_date, start_time, file_name, video_name 
        FROM trainings 
        ORDER BY training_date DESC, start_time DESC
    ");
} else {
    // Untuk trainer, hanya tampilkan training yang dibuat oleh trainer yang login
    $stmt = $koneksi->prepare("
        SELECT id, judul, nama_trainer, training_date, start_time, file_name, video_name 
        FROM trainings 
        WHERE nama_trainer = ? 
        ORDER BY training_date DESC, start_time DESC
    ");
    $stmt->bind_param("s", $nama_user);
}
$stmt->execute();
$training_result = $stmt->get_result();
$stmt->close();

// Set current page untuk highlight menu
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>List Kegiatan Training - E-Learning</title>
    <link rel="icon" type="image/png" href="image/icon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        :root {
            --soft-ui-primary-start: #00C4CC; /* Teal */
            --soft-ui-primary-end: #CB0C9F; /* Purple */
            --soft-ui-secondary: #67748E; /* Secondary color */
            --soft-ui-info: #17C1E8; /* Info color */
            --soft-ui-success: #82D616; /* Success color */
            --soft-ui-warning: #FB8C00; /* Warning color */
            --soft-ui-danger: #EA0606; /* Danger color */
            --soft-ui-light: #E9ECEF; /* Light color */
            --soft-ui-dark: #344767; /* Dark color */
            --soft-ui-white: #FFFFFF;
            --soft-ui-gray-100: #F8F9FA;
            --soft-ui-gray-200: #E9ECEF;
            --soft-ui-gray-300: #DEE2E6;
            --soft-ui-gray-400: #CED4DA;
            --soft-ui-gray-500: #ADB5BD;
            --soft-ui-gray-600: #6C757D;
            --soft-ui-gray-700: #495057;
            --soft-ui-gray-800: #343A40;
            --soft-ui-gray-900: #212529;
        }

        html, body {
            height: 100%;
            margin: 0;
            font-family: 'Open Sans', sans-serif;
            background-color: var(--soft-ui-gray-100);
        }
        
        body {
            overflow-x: hidden;
            display: flex;
            flex-direction: column;
        }
        
        .navbar-top {
            position: fixed;
            top: 0;
            width: 100%;
            z-index: 1100;
            background: var(--soft-ui-white);
            border-bottom: 1px solid var(--soft-ui-gray-200);
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.05);
            padding: 0.75rem 1.5rem;
        }
        
        .navbar-top .container-fluid {
            display: flex;
            align-items: center;
            justify-content: space-between;
            flex-wrap: nowrap;
            position: relative;
        }
        
        /* Logo styling */
        .navbar-logo {
            height: 40px;
            width: auto;
            margin-right: 15px;
        }
        
        /* Center title styling */
        .navbar-title {
            position: absolute;
            left: 50%;
            transform: translateX(-50%);
            font-size: 1.3rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            white-space: nowrap;
            pointer-events: none;
        }

        /* Mobile title styling - 2 lines */
        .navbar-title-mobile {
            display: none;
            font-size: 0.85rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            line-height: 1.2;
            flex: 1;
            margin: 0 10px;
        }

        .navbar-title-mobile .line1 {
            display: block;
        }

        .navbar-title-mobile .line2 {
            display: block;
            font-size: 0.8rem;
        }
        
        .navbar-brand {
            font-size: 1.2rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            display: flex;
            align-items: center;
            z-index: 1;
        }

        .navbar-nav {
            position: relative;
            z-index: 1200;
        }

        .navbar-nav .nav-item .nav-link {
            color: var(--soft-ui-secondary);
            font-weight: 600;
            padding: 0.5rem 1rem;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .nav-item .nav-link:hover {
            color: var(--soft-ui-primary-end);
        }

        .navbar-nav .nav-item .nav-link.active {
            color: var(--soft-ui-primary-end);
        }

        .navbar-nav .dropdown-menu {
            position: absolute;
            top: 100%;
            right: 0;
            left: auto;
            z-index: 1300;
            background-color: var(--soft-ui-white);
            border: 1px solid var(--soft-ui-gray-200);
            border-radius: 0.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            min-width: 160px;
            margin-top: 0.125rem;
        }

        .navbar-nav .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .dropdown-item:hover {
            background-color: var(--soft-ui-gray-200);
            color: var(--soft-ui-dark);
        }

        .navbar-nav .dropdown-item.active {
            background-color: var(--soft-ui-primary-end);
            color: var(--soft-ui-white);
        }
        
        .sidebar {
            min-height: 100vh;
            width: 250px;
            position: fixed;
            top: 0;
            left: 0;
            z-index: 1000;
            background: var(--soft-ui-white);
            color: var(--soft-ui-dark);
            padding-top: 70px;
            transition: all 0.3s ease-in-out;
            border-right: 1px solid var(--soft-ui-gray-200);
            box-shadow: 4px 0 15px rgba(0, 0, 0, 0.05);
            border-radius: 0 1rem 1rem 0;
        }
        
        .content {
            margin-left: 250px;
            margin-top: 0;
            padding: 20px;
            padding-top: 90px;
            transition: all 0.3s ease-in-out;
            flex-grow: 1;
        }
        
        .hamburger-mobile {
            display: none;
            background: none;
            border: none;
            color: var(--soft-ui-dark);
            font-size: 1.5rem;
            cursor: pointer;
            margin-right: 15px;
            transition: all 0.3s ease;
            z-index: 1;
        }
        
        .hamburger-mobile:focus {
            outline: none;
        }
        
        .sidebar-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(0, 0, 0, 0.4);
            z-index: 999;
        }
        
        /* Profile Section Styles */
        .profile-section {
            text-align: center;
            padding: 15px 20px;
            border-bottom: 1px solid var(--soft-ui-gray-200);
            background: var(--soft-ui-white);
            border-radius: 0.75rem;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .profile-photo {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--soft-ui-gray-300);
            margin: 0 auto 10px;
            display: block;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: var(--soft-ui-gray-400);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            border: 2px solid var(--soft-ui-gray-300);
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder i {
            font-size: 2rem;
            color: var(--soft-ui-white);
        }
        
        .profile-name {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--soft-ui-dark);
            margin-bottom: 5px;
            word-wrap: break-word;
        }
        
        .profile-info {
            font-size: 0.9rem;
            font-weight: 400;
            color: var(--soft-ui-secondary);
            margin-bottom: 3px;
        }
        
        /* Sidebar Menu Styles */
        .sidebar .nav-link {
            color: var(--soft-ui-secondary);
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
            transition: all 0.2s ease-in-out;
            font-weight: 600;
            display: flex;
            align-items: center;
        }
        
        .sidebar .nav-link:hover {
            background: linear-gradient(90deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-dark);
        }
        
        .sidebar .nav-link.active {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white) !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .sidebar .nav-link.active i {
            color: var(--soft-ui-white);
        }
        
        .sidebar .nav-link i {
            font-size: 1.2rem;
            margin-right: 10px;
            color: var(--soft-ui-secondary);
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-menu {
            background-color: var(--soft-ui-white);
            border: none;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            border-radius: 0.5rem;
            padding: 0.5rem 0;
            margin-top: 0.25rem;
        }

        .sidebar .nav-item.dropdown .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-item:hover {
            background: linear-gradient(90deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-dark);
        }

        .sidebar .nav-item.dropdown .dropdown-item.active {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
        }
        
        /* Training Cards */
        .table-card {
            transition: all 0.3s ease;
            border: none;
            border-radius: 0.75rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            overflow: hidden;
            background: var(--soft-ui-white);
            backdrop-filter: blur(10px);
        }
        
        .table-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.08);
        }
        
        .table-header {
            padding: 25px;
            color: var(--soft-ui-white);
            text-align: center;
            font-weight: 600;
            font-size: 1.25rem;
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
        }
        
        .table-body {
            padding: 25px;
            background: var(--soft-ui-white);
        }
        
        .training-card {
            transition: all 0.3s ease;
            border: none;
            border-radius: 0.75rem;
            overflow: hidden;
            background: var(--soft-ui-white);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            backdrop-filter: blur(10px);
        }
        
        .training-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.08);
        }
        
        .training-card .card-header {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end)) !important;
            border-bottom: none;
            padding: 1.5rem;
        }
        
        .training-card .card-body {
            padding: 1.5rem;
            background: var(--soft-ui-white);
            display: flex;
            flex-direction: column;
        }
        
        .training-card .card-title {
            font-size: 1.2rem;
            font-weight: 600;
            margin-bottom: 0.8rem;
            color: var(--soft-ui-white);
        }
        
        .training-card p {
            font-size: 0.95rem;
            line-height: 1.6;
            color: var(--soft-ui-dark);
            margin-bottom: 0.5rem;
        }
        
        .training-card .badge {
            font-size: 0.8rem;
            padding: 0.5rem;
            border-radius: 0.5rem;
            display: block;
            text-align: center;
            width: 100%;
            box-sizing: border-box;
        }
        
        .training-card .btn-group .btn {
            border-radius: 0;
        }
        
        .training-card .btn-group .btn:first-child {
            border-top-left-radius: 0.5rem;
            border-bottom-left-radius: 0.5rem;
        }
        
        .training-card .btn-group .btn:last-child {
            border-top-right-radius: 0.5rem;
            border-bottom-right-radius: 0.5rem;
        }
        
        .training-card .btn-outline-info {
            padding: 0.5rem;
            font-size: 0.8rem;
            display: flex;
            align-items: center;
            justify-content: center;
            text-align: center;
            width: 100%;
            box-sizing: border-box;
            min-height: 38px;
            border: 1px solid var(--soft-ui-info);
            color: var(--soft-ui-info);
            background: transparent;
            transition: all 0.2s ease-in-out;
        }

        .training-card .btn-outline-info:hover {
            background: var(--soft-ui-info);
            color: var(--soft-ui-white);
            transform: translateY(-2px);
        }
        
        .training-card .file-video-section {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .training-card .file-video-section .row {
            display: flex;
            flex-wrap: nowrap;
            gap: 0.5rem;
        }
        
        .training-card .file-video-section .col-6 {
            flex: 1;
            display: flex;
            flex-direction: column;
            align-items: stretch;
        }
        
        /* Button Styles */
        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            border-radius: 0.5rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
            border: none;
        }
        
        .btn-primary {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
            box-shadow: 0 4px 12px rgba(0, 196, 204, 0.3);
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 196, 204, 0.4);
            color: var(--soft-ui-white);
            background: linear-gradient(195deg, #009ea6, #a80a85);
        }
        
        .btn-success {
            background: linear-gradient(195deg, var(--soft-ui-success), #66bb6a);
            color: var(--soft-ui-white);
            box-shadow: 0 4px 12px rgba(130, 214, 22, 0.3);
        }
        
        .btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(130, 214, 22, 0.4);
            color: var(--soft-ui-white);
        }
        
        .btn-info {
            background: linear-gradient(195deg, var(--soft-ui-info), #29b6f6);
            color: var(--soft-ui-white);
            box-shadow: 0 4px 12px rgba(23, 193, 232, 0.3);
        }
        
        .btn-info:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(23, 193, 232, 0.4);
            color: var(--soft-ui-white);
        }
        
        .btn-secondary {
            background: var(--soft-ui-gray-200);
            color: var(--soft-ui-dark);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .btn-secondary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.15);
            color: var(--soft-ui-dark);
        }
        
        .btn-danger {
            background: linear-gradient(195deg, var(--soft-ui-danger), #f44336);
            color: var(--soft-ui-white);
            box-shadow: 0 4px 12px rgba(234, 6, 6, 0.3);
        }
        
        .btn-danger:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(234, 6, 6, 0.4);
            color: var(--soft-ui-white);
        }
        
        .btn-warning {
            background: linear-gradient(195deg, var(--soft-ui-warning), #ffc107);
            color: var(--soft-ui-dark);
            box-shadow: 0 4px 12px rgba(251, 140, 0, 0.3);
        }
        
        .btn-warning:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(251, 140, 0, 0.4);
            color: var(--soft-ui-dark);
        }
        
        /* Alert Styles */
        .alert {
            border-radius: 0.5rem;
            padding: 1rem;
            border: none;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            background: var(--soft-ui-white);
            backdrop-filter: blur(10px);
        }
        
        .alert-success {
            background: linear-gradient(195deg, rgba(130, 214, 22, 0.1), rgba(102, 187, 106, 0.1));
            color: var(--soft-ui-success);
            border-left: 4px solid var(--soft-ui-success);
        }
        
        .alert-danger {
            background: linear-gradient(195deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-primary-start);
            border-left: 4px solid var(--soft-ui-primary-start);
        }
        
        .alert-info {
            background: linear-gradient(195deg, rgba(23, 193, 232, 0.1), rgba(41, 182, 246, 0.1));
            color: var(--soft-ui-info);
            border-left: 4px solid var(--soft-ui-info);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 700;
            color: var(--soft-ui-dark);
            margin-bottom: 30px;
            text-align: center;
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        footer {
            background: var(--soft-ui-white);
            color: var(--soft-ui-secondary);
            border-top: 1px solid var(--soft-ui-gray-200);
            padding: 1rem 0;
            text-align: center;
            flex-shrink: 0;
            margin-top: 2rem;
        }
        
        .animate-fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Modal Styles */
        .modal-content {
            background: var(--soft-ui-white);
            border-radius: 0.75rem;
            border: none;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            backdrop-filter: blur(10px);
        }
        
        .modal-header {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
            border-radius: 0.75rem 0.75rem 0 0;
            border-bottom: none;
        }
        
        .qr-code-image {
            max-width: 100%;
            width: 256px;
            height: 256px;
            margin: 0 auto;
            display: block;
            border: 2px solid var(--soft-ui-gray-200);
            border-radius: 0.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            background: var(--soft-ui-white);
        }
        
        #qrCodeContainer {
            min-height: 280px;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            background: var(--soft-ui-white);
        }
        
        .spinner-border {
            margin-bottom: 1rem;
            color: var(--soft-ui-primary-start);
        }
        
        #qrReader {
            width: 100%;
            max-width: 400px;
            margin: 0 auto;
        }
        
        #qrReader video {
            width: 100% !important;
            height: auto !important;
            border-radius: 0.5rem;
        }
        
        /* Video Modal Styles */
        .video-modal .modal-content {
            background: var(--soft-ui-dark);
            border: none;
            border-radius: 0.75rem;
        }
        
        .video-modal .modal-body {
            padding: 0;
        }
        
        .video-modal video {
            width: 100%;
            height: auto;
            border-radius: 0 0 0.75rem 0.75rem;
        }
        
        .video-modal .modal-header {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            border-bottom: none;
            padding: 1rem;
            border-radius: 0.75rem 0.75rem 0 0;
        }
        
        .video-modal .modal-title {
            color: var(--soft-ui-white);
            font-size: 1.2rem;
            font-weight: 600;
        }
        
        /* Icon Colors - Soft UI Theme */
        .text-primary { color: var(--soft-ui-primary-start) !important; }
        .text-success { color: var(--soft-ui-success) !important; }
        .text-warning { color: var(--soft-ui-warning) !important; }
        .text-info { color: var(--soft-ui-info) !important; }
        .text-danger { color: var(--soft-ui-danger) !important; }
        
        /* Badge Colors - Soft UI Theme */
        .badge {
            border-radius: 0.5rem !important;
            padding: 0.5rem 1rem !important;
            font-weight: 500 !important;
            min-width: 100px;
            white-space: normal;
            word-wrap: break-word;
            display: inline-block;
            text-align: center;
        }
        
        .bg-secondary {
            background: var(--soft-ui-gray-200) !important;
            color: var(--soft-ui-dark) !important;
        }

        .btn-close-white {
            filter: invert(1) grayscale(100%) brightness(200%);
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .sidebar {
                margin-left: -250px;
                border-radius: 0;
                padding-top: 90px;
            }
            .sidebar-active {
                margin-left: 0;
            }
            .content {
                margin-left: 0;
                padding-top: 100px;
            }
            
            .navbar-title {
                display: none;
            }
            .navbar-logo {
                display: none;
            }
            
            .navbar-title-mobile {
                display: block;
            }
            
            .hamburger-mobile {
                display: block;
            }
            .navbar-nav {
                position: static;
            }
            .navbar-nav .dropdown-menu {
                position: absolute;
                right: 0;
                left: auto;
                z-index: 1300;
            }
            .training-card {
                margin-bottom: 1rem;
            }
            .page-title {
                font-size: 1.8rem;
            }
            
            .navbar-top {
                padding: 0.5rem 1rem;
                min-height: 70px;
            }

            .training-card .card-body {
                padding: 1.2rem;
            }
            .training-card .card-header {
                padding: 1.2rem;
            }
            .training-card .card-title {
                font-size: 1.1rem;
            }
            .training-card p {
                font-size: 0.9rem;
            }
            .training-card .badge,
            .training-card .btn-outline-info {
                font-size: 0.8rem;
                padding: 0.5rem 1rem;
                min-height: 36px;
                min-width: 90px;
                white-space: normal;
                word-wrap: break-word;
            }
            .training-card .file-video-section .row {
                gap: 0.3rem;
            }

            .profile-photo, .profile-photo-placeholder {
                width: 60px;
                height: 60px;
            }
            .profile-photo-placeholder i {
                font-size: 1.5rem;
            }
            .profile-name {
                font-size: 1rem;
            }
            .profile-info {
                font-size: 0.8rem;
            }
        }

        @media (max-width: 576px) {
            .navbar-top {
                padding: 0.5rem 1rem;
                min-height: 70px;
            }
            .navbar-title-mobile {
                font-size: 0.8rem;
            }
            .navbar-title-mobile .line2 {
                font-size: 0.75rem;
            }
            .page-title {
                font-size: 1.5rem;
            }

            .training-card .badge,
            .training-card .btn-outline-info {
                font-size: 0.75rem;
                padding: 0.4rem 0.8rem;
                min-height: 32px;
                min-width: 80px;
                white-space: normal;
                word-wrap: break-word;
            }
            .training-card .file-video-section .row {
                gap: 0.2rem;
            }

            .profile-photo, .profile-photo-placeholder {
                width: 50px;
                height: 50px;
            }
            .profile-photo-placeholder i {
                font-size: 1.2rem;
            }
            .profile-name {
                font-size: 0.9rem;
            }
            .profile-info {
                font-size: 0.7rem;
            }
        }
    </style>
</head>
<body>
    <!-- Top Navbar -->
    <nav class="navbar navbar-expand-lg navbar-top">
        <div class="container-fluid">
            <button class="hamburger-mobile d-md-none" type="button" id="sidebarToggle">
                <i class="bi bi-list"></i>
            </button>
            <a class="navbar-brand d-none d-md-flex" href="#">
                <img src="image/logopim.png" alt="Logo PIM" class="navbar-logo">
            </a>
            <div class="navbar-title d-none d-md-block">
                E-Learning System Pondok Indah Mall
            </div>
            <div class="navbar-title-mobile d-md-none">
                <span class="line1">E-Learning System</span>
                <span class="line2">Pondok Indah Mall</span>
            </div>
            <?php if ($role != 'iss'): ?>
            <ul class="navbar-nav ms-auto mb-2 mb-lg-0">
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle text-dark" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-person-circle"></i>
                    </a>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><a class="dropdown-item <?php echo ($current_page == 'profile.php') ? 'active' : ''; ?>" href="profile.php">Profile</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="logout.php">Logout</a></li>
                    </ul>
                </li>
            </ul>
            <?php endif; ?>
        </div>
    </nav>

    <!-- Sidebar overlay (for mobile) -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Sidebar -->
    <div class="sidebar border-end" id="sidebar">
        <!-- Profile Section moved here, directly after sidebar padding-top -->
        <div class="profile-section">
            <?php if (!empty($user['foto_profil'])): ?>
                <img src="Uploads/profiles/<?php echo htmlspecialchars($user['foto_profil']); ?>" alt="Foto Profil" class="profile-photo">
            <?php else: ?>
                <div class="profile-photo-placeholder">
                    <i class="bi bi-person-circle"></i>
                </div>
            <?php endif; ?>
            <div class="profile-name"><?php echo htmlspecialchars($user['nama_lengkap']); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($user['jabatan'] ?? 'Tidak ada jabatan'); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($user['divisi'] ?? 'Tidak ada divisi'); ?></div>
        </div>
        <div class="p-3">
            <h5 class="mb-4 text-dark">Menu</h5>
            <ul class="nav flex-column">
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'dashboard.php') ? 'active' : ''; ?>" href="dashboard.php">
                        <i class="bi bi-house-door-fill"></i> Dashboard
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'materi.php') ? 'active' : ''; ?>" href="materi.php">
                        <i class="bi bi-book"></i> Materi
                    </a>
                </li>
                <!-- Post Test Menu -->
                <?php if ($role == 'trainer'): ?>
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_posttest.php' || $current_page == 'list_posttest.php' || $current_page == 'view_posttest_scores.php') ? 'active' : ''; ?>" href="#" id="postTestDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="bi bi-clipboard-check"></i> Post Test
                        </a>
                        <ul class="dropdown-menu" aria-labelledby="postTestDropdown">
                            <li><a class="dropdown-item <?php echo ($current_page == 'add_posttest.php') ? 'active' : ''; ?>" href="add_posttest.php">Tambah Post Test</a></li>
                            <li><a class="dropdown-item <?php echo ($current_page == 'list_posttest.php' || $current_page == 'view_posttest_scores.php') ? 'active' : ''; ?>" href="list_posttest.php">List Post Test</a></li>
                        </ul>
                    </li>
                <?php elseif ($role == 'peserta' || $role == 'iss'): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'list_posttest.php' || $current_page == 'take_posttest.php' || $current_page == 'post_test_result.php') ? 'active' : ''; ?>" href="list_posttest.php">
                            <i class="bi bi-clipboard-check"></i> Post Test
                        </a>
                    </li>
                <?php endif; ?>
                <!-- End Post Test Menu -->
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_training.php' || $current_page == 'list_training.php') ? 'active' : ''; ?>" 
                       href="#" id="trainingDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-journal-text"></i> Training
                    </a>
                    <ul class="dropdown-menu" aria-labelledby="trainingDropdown">
                        <?php if($role == 'trainer'): ?>
                            <li><a class="dropdown-item <?php echo ($current_page == 'add_training.php') ? 'active' : ''; ?>" href="add_training.php">Tambah Kegiatan Training</a></li>
                        <?php endif; ?>
                        <li><a class="dropdown-item <?php echo ($current_page == 'list_training.php') ? 'active' : ''; ?>" href="list_training.php">List Training</a></li>
                    </ul>
                </li>
                
                <!-- Tambahkan menu Report dan Data Evaluasi untuk Trainer -->
                <?php if ($_SESSION['role'] == 'trainer'): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'report.php') ? 'active' : ''; ?>" href="report.php">
                            <i class="bi bi-file-earmark-text"></i> Report
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'data_evaluasi.php') ? 'active' : ''; ?>" href="data_evaluasi.php">
                            <i class="bi bi-clipboard-data"></i> Data Evaluasi
                        </a>
                    </li>
                <?php endif; ?>
            </ul>
        </div>
    </div>

    <!-- Content -->
    <div class="content">
        <div class="container-fluid">
            <h2 class="page-title animate-fade-in">Daftar Kegiatan Training</h2>
            <div class="row">
                <div class="col-md-12">
                    <div class="card table-card animate-fade-in">
                        <div class="table-header">
                            <i class="bi bi-journal-text me-2"></i>Daftar Kegiatan Training
                        </div>
                        <div class="table-body">
                            <?php if(isset($_SESSION['error'])): ?>
                                <div class="alert alert-danger">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <?php echo htmlspecialchars($_SESSION['error']); ?>
                                </div>
                                <?php unset($_SESSION['error']); ?>
                            <?php elseif(isset($_SESSION['success'])): ?>
                                <div class="alert alert-success">
                                    <i class="bi bi-check-circle me-2"></i>
                                    <?php echo htmlspecialchars($_SESSION['success']); ?>
                                </div>
                                <?php unset($_SESSION['success']); ?>
                            <?php endif; ?>
                            <?php if(!empty($error)): ?>
                                <div class="alert alert-danger">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <?php echo htmlspecialchars($error); ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if(!empty($success)): ?>
                                <div class="alert alert-success">
                                    <i class="bi bi-check-circle me-2"></i>
                                    <?php echo htmlspecialchars($success); ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if($training_result->num_rows > 0): ?>
                                <div class="row">
                                    <?php 
                                    $no = 1;
                                    $current_datetime = new DateTime();
                                    while($row = $training_result->fetch_assoc()): 
                                        // Format tanggal dan waktu
                                        $training_date = date('d-m-Y', strtotime($row['training_date']));
                                        $start_time = date('H:i', strtotime($row['start_time']));
                                        // Path video
                                        $video_path = $row['video_name'] ? 'Uploads/videos/' . $row['video_name'] : null;
                                        // Path file
                                        $file_path = $row['file_name'] ? 'Uploads/trainings/' . $row['file_name'] : null;
                                        // Cek keberadaan file
                                        $file_exists = $file_path && file_exists($file_path);
                                        // ✅ Cek apakah user sudah bergabung (untuk peserta dan ISS)
                                        $has_joined = false;
                                        $is_expired = false;
                                        if ($role == 'peserta' || $role == 'iss') {
                                            // Cek apakah sudah bergabung dengan user_type
                                            $stmt = $koneksi->prepare("SELECT id FROM training_participants WHERE training_id = ? AND user_id = ? AND user_type = ?");
                                            $stmt->bind_param("iis", $row['id'], $user_id, $role);
                                            $stmt->execute();
                                            $join_check = $stmt->get_result();
                                            $has_joined = $join_check->num_rows > 0;
                                            $stmt->close();

                                            // Cek apakah waktu training sudah lewat (1 jam setelah waktu mulai)
                                            $training_datetime = new DateTime($row['training_date'] . ' ' . $row['start_time']);
                                            $training_datetime->add(new DateInterval('PT1H')); // Tambah 1 jam
                                            $is_expired = $current_datetime >= $training_datetime;
                                        }
                                    ?>
                                    <div class="col-lg-6 col-xl-4 mb-4">
                                        <div class="card h-100 shadow-sm training-card position-relative">
                                            <?php if ($role == 'trainer'): ?>
                                                <a href="delete_training.php?id=<?php echo $row['id']; ?>" 
                                                   class="btn btn-danger btn-sm position-absolute top-0 end-0 m-2 rounded-circle d-flex align-items-center justify-content-center" 
                                                   style="width: 30px; height: 30px; z-index: 10;" 
                                                   onclick="return confirm('Yakin ingin menghapus training ini?')">
                                                    <i class="bi bi-trash"></i>
                                                </a>
                                            <?php endif; ?>
                                            <div class="card-header bg-gradient text-white">
                                                <h5 class="card-title mb-1"><?php echo htmlspecialchars($row['judul']); ?></h5>
                                                <small class="opacity-75"><i class="bi bi-hash"></i>Training <?php echo $no++; ?></small>
                                            </div>
                                            <div class="card-body d-flex flex-column">
                                                <div class="mb-3">
                                                    <p class="mb-2"><i class="bi bi-person-badge text-primary me-2"></i><strong>Trainer:</strong> <?php echo htmlspecialchars($row['nama_trainer']); ?></p>
                                                    <p class="mb-2"><i class="bi bi-calendar-event text-success me-2"></i><strong>Tanggal:</strong> <?php echo htmlspecialchars($training_date); ?></p>
                                                    <p class="mb-2"><i class="bi bi-clock text-warning me-2"></i><strong>Jam Mulai:</strong> <?php echo htmlspecialchars($start_time); ?></p>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <div class="row">
                                                        <div class="col-6">
                                                            <p class="mb-1"><strong><i class="bi bi-file-earmark me-1"></i>File:</strong></p>
                                                            <?php if ($file_exists): ?>
                                                                <a href="view.php?id=<?php echo $row['id']; ?>" 
                                                                   class="btn btn-sm btn-outline-info w-100" 
                                                                   title="Lihat File">
                                                                    <i class="bi bi-file-earmark"></i> Lihat File
                                                                </a>
                                                            <?php else: ?>
                                                                <span class="badge bg-secondary">Tidak ada file</span>
                                                            <?php endif; ?>
                                                        </div>
                                                        <div class="col-6">
                                                            <p class="mb-1"><strong><i class="bi bi-camera-video me-1"></i>Video:</strong></p>
                                                            <?php if($video_path && file_exists($video_path)): ?>
                                                                <a href="#" 
                                                                   class="btn btn-sm btn-outline-info w-100 play-video-btn" 
                                                                   data-video-path="<?php echo htmlspecialchars($video_path); ?>"
                                                                   data-video-title="<?php echo htmlspecialchars($row['judul']); ?>"
                                                                   data-bs-toggle="modal"
                                                                   data-bs-target="#videoModal"
                                                                   title="Lihat Video">
                                                                    <i class="bi bi-camera-video"></i> Lihat Video
                                                                </a>
                                                            <?php else: ?>
                                                                <span class="badge bg-secondary">No video</span>
                                                            <?php endif; ?>
                                                        </div>
                                                    </div>
                                                </div>
                                                
                                                <div class="mt-auto">
                                                    <?php if($role == 'peserta' || $role == 'iss'): ?>
                                                        <?php if($has_joined): ?>
                                                            <!-- Jika sudah join, tampilkan button Join (disabled) dan button Evaluasi -->
                                                            <div class="btn-group w-100 mb-2" role="group">
                                                                <button class="btn btn-secondary disabled">
                                                                    <i class="bi bi-check-circle"></i> Sudah Join
                                                                </button>
                                                                <a href="evaluasi.php?training_id=<?php echo $row['id']; ?>&training_name=<?php echo urlencode($row['judul']); ?>" class="btn btn-info">
                                                                    <i class="bi bi-clipboard-check"></i> Evaluasi
                                                                </a>
                                                            </div>
                                                        <?php elseif($is_expired): ?>
                                                            <!-- Jika waktu habis, hanya tampilkan button waktu habis -->
                                                            <button class="btn btn-danger w-100 disabled">
                                                                <i class="bi bi-clock"></i> Waktu Habis
                                                            </button>
                                                        <?php else: ?>
                                                            <!-- Jika belum join dan waktu belum habis, tampilkan button Join dan Evaluasi -->
                                                            <div class="btn-group w-100 mb-2" role="group">
                                                                <button class="btn btn-success scan-qr-btn" 
                                                                        data-bs-toggle="modal" 
                                                                        data-bs-target="#scanQrModal" 
                                                                        data-training-id="<?php echo $row['id']; ?>">
                                                                    <i class="bi bi-camera"></i> Join
                                                                </button>
                                                                <a href="evaluasi.php?training_id=<?php echo $row['id']; ?>&training_name=<?php echo urlencode($row['judul']); ?>" class="btn btn-info">
                                                                    <i class="bi bi-clipboard-check"></i> Evaluasi
                                                                </a>
                                                            </div>
                                                        <?php endif; ?>
                                                    <?php elseif($role == 'trainer'): ?>
                                                        <div class="btn-group w-100" role="group">
                                                            <a href="participant.php?training_id=<?php echo $row['id']; ?>" 
                                                               class="btn btn-primary">
                                                                <i class="bi bi-people"></i> Peserta
                                                            </a>
                                                            <button class="btn btn-warning start-qr-btn" 
                                                                    data-bs-toggle="modal" 
                                                                    data-bs-target="#qrCodeModal" 
                                                                    data-training-id="<?php echo $row['id']; ?>">
                                                                <i class="bi bi-qr-code"></i> Start
                                                            </button>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endwhile; ?>
                                </div>
                                <?php else: ?>
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle me-2"></i> 
                                        Belum ada kegiatan training yang tersedia.
                                        <?php if($role == 'trainer'): ?>
                                            <a href="add_training.php" class="alert-link">Tambah kegiatan sekarang</a>.
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- QR Code Modal (Trainer) -->
    <div class="modal fade" id="qrCodeModal" tabindex="-1" aria-labelledby="qrCodeModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="qrCodeModalLabel"><i class="bi bi-qr-code me-2"></i>QR Code untuk Training</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center">
                    <div id="qrCodeContainer">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <p class="mt-2">Memuat QR Code...</p>
                    </div>
                    <p class="mt-3" id="qrCodeDescription" style="display: none;">
                        <strong><i class="bi bi-info-circle me-2"></i>Instruksi:</strong><br>
                        Peserta dapat memindai QR Code ini untuk bergabung dengan training.<br>
                        <small class="text-muted">QR Code ini berisi link untuk join training</small>
                    </p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle me-2"></i>Tutup
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Scan QR Modal (Peserta) -->
    <div class="modal fade" id="scanQrModal" tabindex="-1" aria-labelledby="scanQrModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="scanQrModalLabel"><i class="bi bi-camera me-2"></i>Pindai QR Code</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center">
                    <div id="qrReader"></div>
                    <p class="mt-3"><i class="bi bi-info-circle me-2"></i>Arahkan kamera ke QR Code untuk bergabung.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle me-2"></i>Tutup
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Video Player Modal -->
    <div class="modal fade video-modal" id="videoModal" tabindex="-1" aria-labelledby="videoModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="videoModalLabel"><i class="bi bi-camera-video me-2"></i>Video Training</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <video id="videoPlayer" class="w-100" controls>
                        <source src="/placeholder.svg" type="video/mp4">
                        Browser Anda tidak mendukung tag video.
                    </video>
                </div>
            </div>
        </div>
    </div>
    
    <footer class="text-center">
        <div class="container">
            <p class="mb-0"><i class="bi bi-c-circle me-2"></i><?php echo date('Y'); ?> E-Learning System by PIM</p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/qrcode/1.5.3/qrcode.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>
    <script>
        // Toggle sidebar for mobile
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebarToggle');
            const sidebar = document.getElementById('sidebar');
            const sidebarOverlay = document.getElementById('sidebarOverlay');
            const trainingDropdownToggle = document.getElementById('trainingDropdown');
            const postTestDropdownToggle = document.getElementById('postTestDropdown');
            
            let isDropdownClick = false;

            function toggleSidebar() {
                sidebar.classList.toggle('sidebar-active');
                sidebarOverlay.style.display = sidebar.classList.contains('sidebar-active') ? 'block' : 'none';
                
                if (sidebar.classList.contains('sidebar-active')) {
                    document.body.style.overflow = 'hidden';
                } else {
                    document.body.style.overflow = '';
                }
            }

            sidebarToggle.addEventListener('click', function(e) {
                e.stopPropagation();
                toggleSidebar();
            });

            sidebarOverlay.addEventListener('click', function() {
                if (!isDropdownClick) {
                    toggleSidebar();
                }
                isDropdownClick = false;
            });

            if (trainingDropdownToggle) {
                trainingDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }

            if (postTestDropdownToggle) {
                postTestDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }
            
            document.addEventListener('click', function(e) {
                if (window.innerWidth <= 768) {
                    let clickedInsideAnyDropdown = false;
                    if (trainingDropdownToggle && trainingDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (postTestDropdownToggle && postTestDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (!clickedInsideAnyDropdown) {
                        isDropdownClick = false;
                    }
                }
            });

            const regularNavLinks = document.querySelectorAll('.sidebar .nav-link:not(.dropdown-toggle)');
            regularNavLinks.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            const dropdownItems = document.querySelectorAll('.sidebar .dropdown-menu .dropdown-item');
            dropdownItems.forEach(item => {
                item.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            window.addEventListener('resize', function() {
                if (window.innerWidth > 768) {
                    sidebar.classList.remove('sidebar-active');
                    sidebarOverlay.style.display = 'none';
                    document.body.style.overflow = '';
                    isDropdownClick = false;
                }
            });
        
            // Handle Start QR Code (Trainer) dengan API Online
            document.addEventListener('click', function(e) {
                if (e.target.closest('.start-qr-btn')) {
                    e.preventDefault();
                    const button = e.target.closest('.start-qr-btn');
                    const trainingId = button.getAttribute('data-training-id');
                    
                    console.log('Start QR button clicked, training ID:', trainingId);
                    
                    // Tunggu modal terbuka sepenuhnya
                    const qrModal = document.getElementById('qrCodeModal');
                    
                    qrModal.addEventListener('shown.bs.modal', function() {
                        generateQRCode(trainingId);
                    }, { once: true });
                }
            });
            
            function generateQRCode(trainingId) {
                const qrContainer = document.getElementById('qrCodeContainer');
                
                if (!qrContainer) {
                    console.error('QR Container not found');
                    return;
                }
                
                // Clear container
                qrContainer.innerHTML = '';
                
                // Buat URL untuk join
                const joinUrl = window.location.origin + window.location.pathname + '?join_id=' + trainingId;
                console.log('Generating QR for URL:', joinUrl);
                
                // Buat element img untuk QR Code menggunakan API online
                const qrImg = document.createElement('img');
                qrImg.className = 'qr-code-image';
                qrImg.alt = 'QR Code untuk Training';
                
                // Gunakan API QR Server untuk generate QR Code
                const qrApiUrl = `https://api.qrserver.com/v1/create-qr-code/?size=256x256&data=${encodeURIComponent(joinUrl)}&format=png&margin=10&bgcolor=255-255-255&color=0-0-0`;
                
                qrImg.onload = function() {
                    console.log('QR Code loaded successfully');
                    // Tampilkan deskripsi setelah QR berhasil dimuat
                    const description = document.getElementById('qrCodeDescription');
                    if (description) {
                        description.style.display = 'block';
                    }
                };
                
                qrImg.onerror = function() {
                    console.error('Failed to load QR Code, trying Google Charts API');
                    // Fallback ke Google Charts API
                    const fallbackUrl = `https://chart.googleapis.com/chart?chs=256x256&cht=qr&chl=${encodeURIComponent(joinUrl)}&choe=UTF-8`;
                    qrImg.src = fallbackUrl;
                    
                    qrImg.onerror = function() {
                        console.error('All QR APIs failed');
                        qrContainer.innerHTML = `
                            <div class="alert alert-danger">
                                <h6>Gagal memuat QR Code</h6>
                                <p class="mb-0">Link manual: <a href="${joinUrl}" target="_blank">${joinUrl}</a></p>
                            </div>
                        `;
                    };
                };
                
                qrImg.src = qrApiUrl;
                qrContainer.appendChild(qrImg);
            }
            
            // Handle Scan QR Code (Peserta)
            let html5QrCode;
            const scanModal = document.getElementById('scanQrModal');
            
            if (scanModal) {
                scanModal.addEventListener('shown.bs.modal', function() {
                    const qrReaderElement = document.getElementById('qrReader');
                    if (qrReaderElement && typeof Html5Qrcode !== 'undefined') {
                        html5QrCode = new Html5Qrcode('qrReader');
                        html5QrCode.start(
                            { facingMode: 'environment' },
                            { 
                                fps: 10, 
                                qrbox: { width: 250, height: 250 },
                                aspectRatio: 1.0
                            },
                            (decodedText, decodedResult) => {
                                console.log('QR Code detected:', decodedText);
                                // QR Code detected, redirect to URL
                                html5QrCode.stop().then(() => {
                                    html5QrCode.clear();
                                    window.location.href = decodedText;
                                }).catch(err => {
                                    console.error('Error stopping scanner:', err);
                                    window.location.href = decodedText;
                                });
                            },
                            (errorMessage) => {
                                // Ignore scan errors (e.g., no QR code in view)
                            }
                        ).catch(err => {
                            console.error('Error starting QR scanner:', err);
                            alert('Gagal mengakses kamera. Pastikan Anda menggunakan HTTPS dan memberikan izin kamera.');
                        });
                    } else {
                        console.error('Html5Qrcode library not available or qrReader element not found');
                        const qrReaderElement = document.getElementById('qrReader');
                        if (qrReaderElement) {
                            qrReaderElement.innerHTML = '<div class="alert alert-warning">QR Scanner tidak tersedia. Pastikan library Html5Qrcode ter-load dengan benar.</div>';
                        }
                    }
                });
        
                scanModal.addEventListener('hidden.bs.modal', function() {
                    if (html5QrCode) {
                        html5QrCode.stop().then(() => {
                            html5QrCode.clear();
                        }).catch(err => {
                            console.error('Error stopping QR scanner:', err);
                        });
                    }
                });
            }

            // Handle Video Modal
            const videoModal = document.getElementById('videoModal');
            const videoPlayer = document.getElementById('videoPlayer');
            const videoModalLabel = document.getElementById('videoModalLabel');
            
            document.querySelectorAll('.play-video-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const videoPath = this.getAttribute('data-video-path');
                    const videoTitle = this.getAttribute('data-video-title');
                    
                    // Set video source
                    videoPlayer.querySelector('source').src = videoPath;
                    videoPlayer.load();
                    
                    // Set modal title
                    videoModalLabel.innerHTML = '<i class="bi bi-camera-video me-2"></i>Video: ' + videoTitle;
                });
            });
            
            // Pause video when modal is closed
            videoModal.addEventListener('hidden.bs.modal', function() {
                videoPlayer.pause();
                videoPlayer.currentTime = 0;
            });
        });
    </script>
</body>
</html>