<?php
session_start();
require_once 'koneksi.php';

// Enable error logging
ini_set('log_errors', 1);
ini_set('error_log', dirname(__FILE__) . '/php-error.log');

// Check login
checkLogin();

// Allow both peserta and iss roles
if (!in_array($_SESSION['role'], ['peserta', 'iss'])) {
    header("Location: dashboard.php?error=Akses ditolak");
    exit();
}

$user_id = $_SESSION['user_id'];
$role = $_SESSION['role'];
$nik_karyawan = $_SESSION['nik_karyawan'];

// Ambil informasi user menggunakan prepared statement
if ($role == 'trainer') {
    $stmt = $koneksi->prepare("SELECT nama_lengkap, foto_profil, jabatan, divisi FROM trainers WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
} elseif ($role == 'iss') {
    // ISS table mungkin tidak memiliki kolom foto_profil
    $stmt = $koneksi->prepare("SELECT nama_lengkap, jabatan, divisi FROM iss WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $iss_data = $result->fetch_assoc();
    
    // Set data user dengan foto_profil null untuk ISS
    $user = [
        'nama_lengkap' => $iss_data['nama_lengkap'] ?? 'Unknown',
        'foto_profil' => null, // ISS tidak memiliki foto profil
        'jabatan' => $iss_data['jabatan'] ?? 'Tidak ada jabatan',
        'divisi' => $iss_data['divisi'] ?? 'Tidak ada divisi'
    ];
} else {
    $stmt = $koneksi->prepare("SELECT nama_lengkap, foto_profil, jabatan, divisi FROM peserta WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
}
$nama_user = $user['nama_lengkap'] ?? 'Unknown';
$stmt->close();

// Get current step
$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
if ($step < 1 || $step > 4) $step = 1;

// Get training data from URL if provided
$training_id = isset($_GET['training_id']) ? (int)$_GET['training_id'] : null;
$training_name_url = isset($_GET['training_name']) ? urldecode($_GET['training_name']) : '';

// Initialize session data
if (!isset($_SESSION['eval_data'])) {
    $_SESSION['eval_data'] = [];
}

// Auto-fill training data if training_id provided
if ($training_id && !isset($_SESSION['eval_data']['training_name'])) {
    $stmt = $koneksi->prepare("SELECT judul, nama_trainer, training_date FROM trainings WHERE id = ?");
    $stmt->bind_param("i", $training_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($row = $result->fetch_assoc()) {
        $_SESSION['eval_data']['training_id'] = $training_id;
        $_SESSION['eval_data']['training_name'] = $row['judul'];
        $_SESSION['eval_data']['instructor'] = $row['nama_trainer'];
        
        // Convert date format from YYYY-MM-DD to DD-MM-YYYY
        if ($row['training_date']) {
            $date = DateTime::createFromFormat('Y-m-d', $row['training_date']);
            if ($date) {
                $_SESSION['eval_data']['training_date'] = $date->format('d-m-Y');
            }
        }
    }
    $stmt->close();
}

// Set training name from URL if not set
if ($training_name_url && !isset($_SESSION['eval_data']['training_name'])) {
    $_SESSION['eval_data']['training_name'] = $training_name_url;
}

$message = '';
$error = '';

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'next' || $action === 'submit') {
        // Save current step data
        if ($step === 1) {
            $_SESSION['eval_data']['training_id'] = $_POST['training_id'] ?? null;
            $_SESSION['eval_data']['training_name'] = trim($_POST['training_name'] ?? '');
            $_SESSION['eval_data']['instructor'] = trim($_POST['instructor'] ?? '');
            $_SESSION['eval_data']['location'] = trim($_POST['location'] ?? '');
            $_SESSION['eval_data']['participant_name'] = trim($_POST['participant_name'] ?? '');
            $_SESSION['eval_data']['position'] = trim($_POST['position'] ?? '');
            $_SESSION['eval_data']['work_unit'] = trim($_POST['work_unit'] ?? '');
            
            // Validate and format date
            $date_input = trim($_POST['training_date'] ?? '');
            if (preg_match('/^(\d{1,2})-(\d{1,2})-(\d{4})$/', $date_input, $matches)) {
                $day = (int)$matches[1];
                $month = (int)$matches[2];
                $year = (int)$matches[3];
                
                if (checkdate($month, $day, $year)) {
                    $_SESSION['eval_data']['training_date'] = sprintf('%02d-%02d-%04d', $day, $month, $year);
                } else {
                    $error = 'Tanggal tidak valid';
                }
            } else {
                $error = 'Format tanggal harus DD-MM-YYYY';
            }
            
            // Validate required fields
            $required = ['training_name', 'instructor', 'location', 'participant_name', 'position', 'work_unit', 'training_date'];
            foreach ($required as $field) {
                if (empty($_SESSION['eval_data'][$field])) {
                    $error = 'Semua field wajib diisi';
                    break;
                }
            }
            
        } elseif ($step === 2) {
            $fields = ['fasilitator_persiapan', 'fasilitator_materi', 'waktu_training', 'training_interaktif', 'fasilitas_tempat', 'peralatan_training'];
            foreach ($fields as $field) {
                $value = (int)($_POST[$field] ?? 0);
                if ($value < 1 || $value > 4) {
                    $error = 'Semua pertanyaan harus dijawab dengan skala 1-4';
                    break;
                }
                $_SESSION['eval_data'][$field] = $value;
            }
            
        } elseif ($step === 3) {
            $fields = ['ilmu_baru', 'memahami_tujuan', 'sesuai_harapan', 'bermanfaat', 'kualitas_kerja', 'relevan_pekerjaan'];
            foreach ($fields as $field) {
                $value = (int)($_POST[$field] ?? 0);
                if ($value < 1 || $value > 4) {
                    $error = 'Semua pertanyaan harus dijawab dengan skala 1-4';
                    break;
                }
                $_SESSION['eval_data'][$field] = $value;
            }
            
        } elseif ($step === 4) {
            $_SESSION['eval_data']['saran_masukan'] = trim($_POST['saran_masukan'] ?? '');
        }
        
        // Navigate or submit
        if (!$error) {
            if ($action === 'next' && $step < 4) {
                header("Location: evaluasi.php?step=" . ($step + 1));
                exit();
            } elseif ($action === 'submit' && $step === 4) {
                // Save to database
                try {
                    $data = $_SESSION['eval_data'];
                    
                    // Convert date format for database
                    $date_parts = explode('-', $data['training_date']);
                    $db_date = $date_parts[2] . '-' . $date_parts[1] . '-' . $date_parts[0];
                    
                    $sql = "INSERT INTO evaluasi_training (
                        training_id, user_id, nik_karyawan, training_name, instructor, location, 
                        participant_name, position, work_unit, training_date, fasilitator_persiapan, 
                        fasilitator_materi, waktu_training, training_interaktif, fasilitas_tempat, 
                        peralatan_training, ilmu_baru, memahami_tujuan, sesuai_harapan, bermanfaat, 
                        kualitas_kerja, relevan_pekerjaan, saran_masukan
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    
                    $stmt = $koneksi->prepare($sql);
                    $stmt->bind_param("iissssssssiiiiiiiiiiiis",
                        $data['training_id'],
                        $_SESSION['user_id'],
                        $_SESSION['nik_karyawan'],
                        $data['training_name'],
                        $data['instructor'],
                        $data['location'],
                        $data['participant_name'],
                        $data['position'],
                        $data['work_unit'],
                        $db_date,
                        $data['fasilitator_persiapan'],
                        $data['fasilitator_materi'],
                        $data['waktu_training'],
                        $data['training_interaktif'],
                        $data['fasilitas_tempat'],
                        $data['peralatan_training'],
                        $data['ilmu_baru'],
                        $data['memahami_tujuan'],
                        $data['sesuai_harapan'],
                        $data['bermanfaat'],
                        $data['kualitas_kerja'],
                        $data['relevan_pekerjaan'],
                        $data['saran_masukan']
                    );
                    
                    if ($stmt->execute()) {
                        unset($_SESSION['eval_data']);
                        $message = 'Evaluasi berhasil disimpan!';
                        echo "<script>setTimeout(() => window.location.href = 'dashboard.php?success=Evaluasi berhasil disimpan', 2000);</script>";
                    } else {
                        $error = 'Gagal menyimpan data: ' . $stmt->error;
                    }
                    $stmt->close();
                    
                } catch (Exception $e) {
                    $error = 'Error: ' . $e->getMessage();
                    error_log("Evaluation save error: " . $e->getMessage());
                }
            }
        }
        
    } elseif ($action === 'back' && $step > 1) {
        header("Location: evaluasi.php?step=" . ($step - 1));
        exit();
    }
}

// Helper functions
function getStepTitle($step) {
    $titles = [
        1 => 'Data Training',
        2 => 'Evaluasi Fasilitator & Fasilitas', 
        3 => 'Evaluasi Manfaat & Relevansi',
        4 => 'Saran & Masukan'
    ];
    return $titles[$step] ?? 'Evaluasi Training';
}

function getProgressPercent($step) {
    return ($step / 4) * 100;
}

// Set current page untuk highlight menu
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Evaluasi Training - Step <?= $step ?> - E-Learning</title>
    <link rel="icon" type="image/png" href="image/icon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        :root {
            --soft-ui-primary: #cb0c9f;
            --soft-ui-secondary: #67748e;
            --soft-ui-info: #17c1e8;
            --soft-ui-success: #82d616;
            --soft-ui-warning: #fb8c00;
            --soft-ui-danger: #ea0606;
            --soft-ui-light: #e9ecef;
            --soft-ui-dark: #344767;
            --soft-ui-white: #ffffff;
            --soft-ui-gray-100: #f8f9fa;
            --soft-ui-gray-200: #e9ecef;
            --soft-ui-gray-300: #dee2e6;
            --soft-ui-gray-400: #ced4da;
            --soft-ui-gray-500: #adb5bd;
            --soft-ui-gray-600: #6c757d;
            --soft-ui-gray-700: #495057;
            --soft-ui-gray-800: #343a40;
            --soft-ui-gray-900: #212529;
        }

        html, body {
            height: 100%;
            margin: 0;
            font-family: 'Open Sans', sans-serif;
            background-color: var(--soft-ui-gray-100);
        }
        
        body {
            overflow-x: hidden;
            display: flex;
            flex-direction: column;
        }
        
        .navbar-top {
            position: fixed;
            top: 0;
            width: 100%;
            z-index: 1100;
            background: var(--soft-ui-white);
            border-bottom: 1px solid var(--soft-ui-gray-200);
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.05);
            padding: 0.75rem 1.5rem;
        }
        
        .navbar-top .container-fluid {
            display: flex;
            align-items: center;
            justify-content: space-between;
            flex-wrap: nowrap;
            position: relative;
        }
        
        .navbar-logo {
            height: 40px;
            width: auto;
            margin-right: 15px;
        }
        
        .navbar-title {
            position: absolute;
            left: 50%;
            transform: translateX(-50%);
            font-size: 1.3rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            white-space: nowrap;
            pointer-events: none;
        }

        .navbar-title-mobile {
            display: none;
            font-size: 0.85rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            line-height: 1.2;
            flex: 1;
            margin: 0 10px;
        }

        .navbar-title-mobile .line1 {
            display: block;
        }

        .navbar-title-mobile .line2 {
            display: block;
            font-size: 0.8rem;
        }
        
        .navbar-brand {
            font-size: 1.2rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            display: flex;
            align-items: center;
            z-index: 1;
        }

        .navbar-nav {
            position: relative;
            z-index: 1200;
        }

        .navbar-nav .nav-item .nav-link {
            color: var(--soft-ui-secondary);
            font-weight: 600;
            padding: 0.5rem 1rem;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .nav-item .nav-link:hover {
            color: var(--soft-ui-primary);
        }

        .navbar-nav .nav-item .nav-link.active {
            color: var(--soft-ui-primary);
        }

        .navbar-nav .dropdown-menu {
            position: absolute;
            top: 100%;
            right: 0;
            left: auto;
            z-index: 1300;
            background-color: var(--soft-ui-white);
            border: 1px solid var(--soft-ui-gray-200);
            border-radius: 0.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            min-width: 160px;
            margin-top: 0.125rem;
        }

        .navbar-nav .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .dropdown-item:hover {
            background-color: var(--soft-ui-gray-200);
            color: var(--soft-ui-dark);
        }

        .navbar-nav .dropdown-item.active {
            background-color: var(--soft-ui-primary);
            color: var(--soft-ui-white);
        }
        
        .sidebar {
            min-height: 100vh;
            width: 250px;
            position: fixed;
            top: 0;
            left: 0;
            z-index: 1000;
            background: var(--soft-ui-white);
            color: var(--soft-ui-dark);
            padding-top: 70px;
            transition: all 0.3s ease-in-out;
            border-right: 1px solid var(--soft-ui-gray-200);
            box-shadow: 4px 0 15px rgba(0, 0, 0, 0.05);
            border-radius: 0 1rem 1rem 0;
        }
        
        .content {
            margin-left: 250px;
            margin-top: 0;
            padding: 20px;
            padding-top: 90px;
            transition: all 0.3s ease-in-out;
            flex-grow: 1;
        }
        
        .hamburger-mobile {
            display: none;
            background: none;
            border: none;
            color: var(--soft-ui-dark);
            font-size: 1.5rem;
            cursor: pointer;
            margin-right: 15px;
            transition: all 0.3s ease;
            z-index: 1;
        }
        
        .hamburger-mobile:focus {
            outline: none;
        }
        
        .sidebar-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(0, 0, 0, 0.4);
            z-index: 999;
        }
        
        .profile-section {
            text-align: center;
            padding: 15px 20px;
            border-bottom: 1px solid var(--soft-ui-gray-200);
            background: var(--soft-ui-white);
            border-radius: 0.75rem;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .profile-photo {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--soft-ui-gray-300);
            margin: 0 auto 10px;
            display: block;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: var(--soft-ui-gray-400);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            border: 2px solid var(--soft-ui-gray-300);
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder i {
            font-size: 2rem;
            color: var(--soft-ui-white);
        }
        
        .profile-name {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--soft-ui-dark);
            margin-bottom: 5px;
            word-wrap: break-word;
        }
        
        .profile-info {
            font-size: 0.9rem;
            font-weight: 400;
            color: var(--soft-ui-secondary);
            margin-bottom: 3px;
        }
        
        .sidebar .nav-link {
            color: var(--soft-ui-secondary);
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
            transition: all 0.2s ease-in-out;
            font-weight: 600;
            display: flex;
            align-items: center;
        }
        
        .sidebar .nav-link:hover {
            background-color: var(--soft-ui-gray-200);
            color: var(--soft-ui-dark);
        }
        
        .sidebar .nav-link.active {
            background-color: var(--soft-ui-primary);
            color: var(--soft-ui-white) !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .sidebar .nav-link.active i {
            color: var(--soft-ui-white);
        }
        
        .sidebar .nav-link i {
            font-size: 1.2rem;
            margin-right: 10px;
            color: var(--soft-ui-secondary);
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-menu {
            background-color: var(--soft-ui-white);
            border: none;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            border-radius: 0.5rem;
            padding: 0.5rem 0;
            margin-top: 0.25rem;
        }

        .sidebar .nav-item.dropdown .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-item:hover {
            background-color: var(--soft-ui-gray-200);
            color: var(--soft-ui-dark);
        }

        .sidebar .nav-item.dropdown .dropdown-item.active {
            background-color: var(--soft-ui-primary);
            color: var(--soft-ui-white);
        }
        
        .card {
            background-color: var(--soft-ui-white);
            border-radius: 0.75rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            transition: all 0.2s ease-in-out;
            border: none;
        }

        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.08);
        }
        
        .card-header {
            padding: 1.5rem;
            color: var(--soft-ui-white);
            text-align: center;
            font-weight: 700;
            font-size: 1.25rem;
            background: linear-gradient(195deg, var(--soft-ui-primary), #a80a85);
            border-bottom: none;
            border-radius: 0.75rem 0.75rem 0 0;
        }
        
        .card-body {
            padding: 1.5rem;
            background: var(--soft-ui-white);
        }
        
        .form-label {
            font-weight: 600;
            color: var(--soft-ui-dark);
            margin-bottom: 0.5rem;
        }
        
        .form-control, .form-select {
            border: 1px solid var(--soft-ui-gray-300);
            border-radius: 0.5rem;
            padding: 0.75rem 1rem;
            font-size: 0.875rem;
            transition: all 0.2s ease-in-out;
            background-color: var(--soft-ui-white);
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--soft-ui-primary);
            box-shadow: 0 0 0 0.2rem rgba(203, 12, 159, 0.25);
            background-color: var(--soft-ui-white);
        }
        
        .progress {
            height: 0.75rem;
            border-radius: 0.5rem;
            background-color: var(--soft-ui-gray-200);
            box-shadow: inset 0 1px 2px rgba(0, 0, 0, 0.1);
        }
        
        .progress-bar {
            background: linear-gradient(195deg, var(--soft-ui-primary), #a80a85);
            border-radius: 0.5rem;
            transition: width 0.6s ease;
        }
        
        .form-check-input {
            border-color: var(--soft-ui-gray-400);
            margin-top: 0.25em;
        }

        .form-check-input:checked {
            background-color: var(--soft-ui-primary);
            border-color: var(--soft-ui-primary);
        }

        .form-check-input:focus {
            border-color: var(--soft-ui-primary);
            box-shadow: 0 0 0 0.2rem rgba(203, 12, 159, 0.25);
        }

        .form-check-label {
            font-weight: 500;
            color: var(--soft-ui-dark);
        }
        
        .btn-primary {
            background: linear-gradient(195deg, var(--soft-ui-primary), #a80a85);
            border: none;
            border-radius: 0.5rem;
            padding: 0.75rem 1.5rem;
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--soft-ui-white);
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .btn-primary:hover {
            background: linear-gradient(195deg, #a80a85, var(--soft-ui-primary));
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.15);
            color: var(--soft-ui-white);
        }
        
        .btn-secondary {
            background: var(--soft-ui-gray-200);
            border: none;
            border-radius: 0.5rem;
            padding: 0.75rem 1.5rem;
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--soft-ui-dark);
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .btn-secondary:hover {
            background: var(--soft-ui-gray-300);
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.1);
            color: var(--soft-ui-dark);
        }
        
        .btn-outline-secondary {
            border: 2px solid var(--soft-ui-gray-300);
            color: var(--soft-ui-secondary);
            border-radius: 0.5rem;
            padding: 0.75rem 1.5rem;
            font-weight: 600;
            font-size: 0.875rem;
            transition: all 0.3s ease;
            background: var(--soft-ui-white);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .btn-outline-secondary:hover {
            background: var(--soft-ui-gray-300);
            color: var(--soft-ui-dark);
            border-color: var(--soft-ui-gray-300);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .alert {
            border-radius: 0.75rem;
            padding: 1rem 1.5rem;
            border: none;
            font-weight: 500;
        }

        .alert-success {
            background: linear-gradient(195deg, rgba(130, 214, 22, 0.1), rgba(130, 214, 22, 0.05));
            color: var(--soft-ui-success);
            border-left: 4px solid var(--soft-ui-success);
        }

        .alert-danger {
            background: linear-gradient(195deg, rgba(234, 6, 6, 0.1), rgba(234, 6, 6, 0.05));
            color: var(--soft-ui-danger);
            border-left: 4px solid var(--soft-ui-danger);
        }
        
        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--soft-ui-dark);
            margin-bottom: 2rem;
            text-align: center;
            line-height: 1.2;
        }

        .step-indicator {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
            font-size: 0.875rem;
            color: var(--soft-ui-secondary);
        }

        .step-indicator .step {
            display: flex;
            align-items: center;
            font-weight: 500;
        }

        .step-indicator .step.active {
            color: var(--soft-ui-primary);
            font-weight: 700;
        }

        .evaluation-question {
            background: var(--soft-ui-gray-100);
            border-radius: 0.75rem;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            border-left: 4px solid var(--soft-ui-primary);
            transition: all 0.2s ease-in-out;
        }

        .evaluation-question:hover {
            background: var(--soft-ui-gray-200);
            transform: translateX(5px);
        }

        .evaluation-question .question-text {
            font-weight: 600;
            color: var(--soft-ui-dark);
            margin-bottom: 1rem;
            font-size: 1rem;
        }

        .rating-scale {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 1rem;
        }

        .rating-options {
            display: flex;
            gap: 1.5rem;
        }

        .rating-options .form-check {
            margin-bottom: 0;
        }

        .rating-options .form-check-input {
            width: 1.25rem;
            height: 1.25rem;
        }

        .rating-options .form-check-label {
            font-size: 1rem;
            font-weight: 600;
            margin-left: 0.5rem;
        }

        .scale-label {
            font-size: 0.875rem;
            color: var(--soft-ui-secondary);
            font-weight: 500;
        }
        
        footer {
            background: var(--soft-ui-white);
            color: var(--soft-ui-secondary);
            border-top: 1px solid var(--soft-ui-gray-200);
            padding: 1rem 0;
            text-align: center;
            flex-shrink: 0;
            margin-top: 2rem;
        }
        
        .animate-fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        @media (max-width: 768px) {
            .sidebar {
                margin-left: -250px;
                border-radius: 0;
                padding-top: 90px;
            }
            .sidebar-active {
                margin-left: 0;
            }
            .content {
                margin-left: 0;
                padding-top: 100px;
            }
            
            .navbar-title {
                display: none;
            }
            .navbar-logo {
                display: none;
            }
            
            .navbar-title-mobile {
                display: block;
            }
            
            .hamburger-mobile {
                display: block;
            }
            .navbar-nav {
                position: static;
            }
            .navbar-nav .dropdown-menu {
                position: absolute;
                right: 0;
                left: auto;
                z-index: 1300;
            }
            
            .page-title {
                font-size: 1.8rem;
                text-align: center;
            }

            .rating-options {
                gap: 1rem;
            }

            .rating-options .form-check-input {
                width: 1rem;
                height: 1rem;
            }

            .rating-options .form-check-label {
                font-size: 0.875rem;
            }

            .evaluation-question {
                padding: 1rem;
            }
            
            .navbar-top {
                padding: 0.5rem 1rem;
                min-height: 70px;
            }

            .profile-photo, .profile-photo-placeholder {
                width: 60px;
                height: 60px;
            }
            .profile-photo-placeholder i {
                font-size: 1.5rem;
            }
            .profile-name {
                font-size: 1rem;
            }
            .profile-info {
                font-size: 0.8rem;
            }
        }

        @media (max-width: 576px) {
            .navbar-top {
                padding: 0.5rem 1rem;
                min-height: 70px;
            }
            .navbar-title-mobile {
                font-size: 0.8rem;
            }
            .navbar-title-mobile .line2 {
                font-size: 0.75rem;
            }
            .page-title {
                font-size: 1.5rem;
            }

            .rating-options {
                gap: 0.75rem;
            }

            .scale-label {
                font-size: 0.75rem;
            }

            .evaluation-question .question-text {
                font-size: 0.9rem;
            }

            .profile-photo, .profile-photo-placeholder {
                width: 50px;
                height: 50px;
            }
            .profile-photo-placeholder i {
                font-size: 1.2rem;
            }
            .profile-name {
                font-size: 0.9rem;
            }
            .profile-info {
                font-size: 0.7rem;
            }
        }
    </style>
</head>
<body>
    <!-- Top Navbar -->
    <nav class="navbar navbar-expand-lg navbar-top">
        <div class="container-fluid">
            <button class="hamburger-mobile d-md-none" type="button" id="sidebarToggle">
                <i class="bi bi-list"></i>
            </button>
            <a class="navbar-brand d-none d-md-flex" href="#">
                <img src="image/logopim.png" alt="Logo PIM" class="navbar-logo">
            </a>
            <div class="navbar-title d-none d-md-block">
                E-Learning System Pondok Indah Mall
            </div>
            <div class="navbar-title-mobile d-md-none">
                <span class="line1">E-Learning System</span>
                <span class="line2">Pondok Indah Mall</span>
            </div>
            <?php if ($role != 'iss'): ?>
            <ul class="navbar-nav ms-auto mb-2 mb-lg-0">
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle text-dark" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-person-circle"></i>
                    </a>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><a class="dropdown-item <?php echo ($current_page == 'profile.php') ? 'active' : ''; ?>" href="profile.php">Profile</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="logout.php">Logout</a></li>
                    </ul>
                </li>
            </ul>
            <?php endif; ?>
        </div>
    </nav>

    <!-- Sidebar overlay (for mobile) -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Sidebar -->
    <div class="sidebar border-end" id="sidebar">
        <div class="profile-section">
            <?php if (!empty($user['foto_profil'])): ?>
                <img src="Uploads/profiles/<?php echo htmlspecialchars($user['foto_profil']); ?>" alt="Foto Profil" class="profile-photo">
            <?php else: ?>
                <div class="profile-photo-placeholder">
                    <i class="bi bi-person-circle"></i>
                </div>
            <?php endif; ?>
            <div class="profile-name"><?php echo htmlspecialchars($user['nama_lengkap']); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($user['jabatan'] ?? 'Tidak ada jabatan'); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($user['divisi'] ?? 'Tidak ada divisi'); ?></div>
        </div>
        <div class="p-3">
            <h5 class="mb-4 text-dark">Menu</h5>
            <ul class="nav flex-column">
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'dashboard.php') ? 'active' : ''; ?>" href="dashboard.php">
                        <i class="bi bi-house-door-fill"></i> Dashboard
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'materi.php') ? 'active' : ''; ?>" href="materi.php">
                        <i class="bi bi-book"></i> Materi
                    </a>
                </li>
                <?php if ($role == 'trainer'): ?>
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_posttest.php' || $current_page == 'list_posttest.php' || $current_page == 'view_posttest_scores.php') ? 'active' : ''; ?>" href="#" id="postTestDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="bi bi-clipboard-check"></i> Post Test
                        </a>
                        <ul class="dropdown-menu" aria-labelledby="postTestDropdown">
                            <li><a class="dropdown-item <?php echo ($current_page == 'add_posttest.php') ? 'active' : ''; ?>" href="add_posttest.php">Tambah Post Test</a></li>
                            <li><a class="dropdown-item <?php echo ($current_page == 'list_posttest.php' || $current_page == 'view_posttest_scores.php') ? 'active' : ''; ?>" href="list_posttest.php">List Post Test</a></li>
                        </ul>
                    </li>
                <?php elseif ($role == 'peserta' || $role == 'iss'): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'list_posttest.php' || $current_page == 'take_posttest.php' || $current_page == 'post_test_result.php') ? 'active' : ''; ?>" href="list_posttest.php">
                            <i class="bi bi-clipboard-check"></i> Post Test
                        </a>
                    </li>
                <?php endif; ?>
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_training.php' || $current_page == 'list_training.php') ? 'active' : ''; ?>" href="#" id="trainingDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-journal-text"></i> Training
                    </a>
                    <ul class="dropdown-menu" aria-labelledby="trainingDropdown">
                        <?php if($role == 'trainer'): ?>
                        <li><a class="dropdown-item <?php echo ($current_page == 'add_training.php') ? 'active' : ''; ?>" href="add_training.php">Tambah Kegiatan Training</a></li>
                        <?php endif; ?>
                        <li><a class="dropdown-item <?php echo ($current_page == 'list_training.php') ? 'active' : ''; ?>" href="list_training.php">List Training</a></li>
                    </ul>
                </li>
                <?php if ($role == 'trainer'): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'report.php') ? 'active' : ''; ?>" href="report.php">
                            <i class="bi bi-file-earmark-text"></i> Report
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'data_evaluasi.php') ? 'active' : ''; ?>" href="data_evaluasi.php">
                            <i class="bi bi-clipboard-data"></i> Data Evaluasi
                        </a>
                    </li>
                <?php endif; ?>
            </ul>
        </div>
    </div>

    <!-- Content -->
    <div class="content">
        <div class="container-fluid">
            <h2 class="page-title animate-fade-in">
                <i class="bi bi-clipboard-check me-2"></i>
                Evaluasi Training - Step <?= $step ?>
            </h2>
            
            <!-- Progress -->
            <div class="mb-4 animate-fade-in">
                <div class="progress mb-2">
                    <div class="progress-bar" style="width: <?= getProgressPercent($step) ?>%"></div>
                </div>
                <div class="step-indicator">
                    <span class="step <?= $step == 1 ? 'active' : '' ?>">
                        <i class="bi bi-1-circle me-1"></i>Data Training
                    </span>
                    <span class="step <?= $step == 2 ? 'active' : '' ?>">
                        <i class="bi bi-2-circle me-1"></i>Fasilitator
                    </span>
                    <span class="step <?= $step == 3 ? 'active' : '' ?>">
                        <i class="bi bi-3-circle me-1"></i>Manfaat
                    </span>
                    <span class="step <?= $step == 4 ? 'active' : '' ?>">
                        <i class="bi bi-4-circle me-1"></i>Saran
                    </span>
                </div>
            </div>

            <div class="row">
                <div class="col-12">
                    <div class="card animate-fade-in">
                        <div class="card-header">
                            <i class="bi bi-clipboard-check me-2"></i>
                            <?= getStepTitle($step) ?>
                        </div>
                        <div class="card-body">
                            <?php if ($message): ?>
                                <div class="alert alert-success animate-fade-in">
                                    <i class="bi bi-check-circle me-2"></i><?= htmlspecialchars($message) ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($error): ?>
                                <div class="alert alert-danger animate-fade-in">
                                    <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
                                </div>
                            <?php endif; ?>

                            <form method="POST">
                                <?php if ($step === 1): ?>
                                    <!-- Step 1: Basic Data -->
                                    <input type="hidden" name="training_id" value="<?= htmlspecialchars($_SESSION['eval_data']['training_id'] ?? '') ?>">
                                    
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">
                                                <i class="bi bi-bookmark me-2"></i>Nama Training <span class="text-danger">*</span>
                                            </label>
                                            <input type="text" class="form-control" name="training_name" 
                                                   value="<?= htmlspecialchars($_SESSION['eval_data']['training_name'] ?? '') ?>" required>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">
                                                <i class="bi bi-person-badge me-2"></i>Instruktur <span class="text-danger">*</span>
                                            </label>
                                            <input type="text" class="form-control" name="instructor" 
                                                   value="<?= htmlspecialchars($_SESSION['eval_data']['instructor'] ?? '') ?>" required>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">
                                                <i class="bi bi-geo-alt me-2"></i>Lokasi Training <span class="text-danger">*</span>
                                            </label>
                                            <input type="text" class="form-control" name="location" 
                                                   value="<?= htmlspecialchars($_SESSION['eval_data']['location'] ?? '') ?>" required>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">
                                                <i class="bi bi-calendar me-2"></i>Tanggal Training <span class="text-danger">*</span>
                                            </label>
                                            <input type="text" class="form-control" name="training_date" 
                                                   value="<?= htmlspecialchars($_SESSION['eval_data']['training_date'] ?? '') ?>" 
                                                   placeholder="DD-MM-YYYY" required>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">
                                                <i class="bi bi-person me-2"></i>Nama Peserta <span class="text-danger">*</span>
                                            </label>
                                            <input type="text" class="form-control" name="participant_name" 
                                                   value="<?= htmlspecialchars($_SESSION['eval_data']['participant_name'] ?? '') ?>" required>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">
                                                <i class="bi bi-briefcase me-2"></i>Jabatan <span class="text-danger">*</span>
                                            </label>
                                            <select class="form-select" name="position" required>
                                                <option value="">-- Pilih Jabatan --</option>
                                                <option value="Pelaksana" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Pelaksana') ? 'selected' : '' ?>>Pelaksana</option>
                                                <option value="Leader" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Leader') ? 'selected' : '' ?>>Leader</option>
                                                <option value="Staff" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Staff') ? 'selected' : '' ?>>Staff</option>
                                                <option value="Asst.Supervisor" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Asst.Supervisor') ? 'selected' : '' ?>>Asst.Supervisor</option>
                                                <option value="Supervisor" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Supervisor') ? 'selected' : '' ?>>Supervisor</option>
                                                <option value="Asst.Manager" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Asst.Manager') ? 'selected' : '' ?>>Asst.Manager</option>
                                                <option value="Manager" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Manager') ? 'selected' : '' ?>>Manager</option>
                                                <option value="Deputy GM" <?= (($_SESSION['eval_data']['position'] ?? '') === 'Deputy GM') ? 'selected' : '' ?>>Deputy GM</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">
                                                <i class="bi bi-building me-2"></i>Unit Kerja <span class="text-danger">*</span>
                                            </label>
                                            <select class="form-select" name="work_unit" required> 
                                                <option value="">-- Unit Kerja --</option>
                                                <option value="HR & GA" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'HR & GA') ? 'selected' : '' ?>>HR & GA</option>
                                                <option value="HSE & CCTV" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'HSE & CCTV') ? 'selected' : '' ?>>HSE & CCTV</option>
                                                <option value="Finance" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Finance') ? 'selected' : '' ?>>Finance</option>
                                                <option value="Marketing" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Marketing') ? 'selected' : '' ?>>Marketing</option>
                                                <option value="Promosi" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Promosi') ? 'selected' : '' ?>>Promosi</option>
                                                <option value="Customer Service" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Customer Service') ? 'selected' : '' ?>>Customer Service</option>
                                                <option value="Legal" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Legal') ? 'selected' : '' ?>>Legal</option>
                                                <option value="Engineering" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Engineering') ? 'selected' : '' ?>>Engineering</option>
                                                <option value="Engineering - BM" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Engineering - BM') ? 'selected' : '' ?>>Engineering - BM</option>
                                                <option value="Engineering - ME" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Engineering - ME') ? 'selected' : '' ?>>Engineering - ME</option>
                                                <option value="Engineering - Fit Out" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Engineering - Fit Out') ? 'selected' : '' ?>>Engineering - Fit Out</option>
                                                <option value="Engineering - Landscape" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Engineering - Landscape') ? 'selected' : '' ?>>Engineering - Landscape</option>
                                                <option value="Engineering - Waterpark" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Engineering - Waterpark') ? 'selected' : '' ?>>Engineering - Waterpark</option>
                                                <option value="Security" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Security') ? 'selected' : '' ?>>Security</option>
                                                <option value="Parkir" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Parkir') ? 'selected' : '' ?>>Parkir</option>
                                                <option value="Cleaner" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'Cleaner') ? 'selected' : '' ?>>Cleaner</option>
                                                <option value="IT" <?= (($_SESSION['eval_data']['work_unit'] ?? '') === 'IT') ? 'selected' : '' ?>>IT</option>
                                            </select>
                                        </div>
                                    </div>

                                <?php elseif ($step === 2): ?>
                                    <!-- Step 2: Facilitator & Facilities -->
                                    <div class="alert alert-info mb-4">
                                        <i class="bi bi-info-circle me-2"></i>
                                        Berikan penilaian dengan skala 1-4 (1=Sangat Tidak Setuju, 4=Sangat Setuju)
                                    </div>
                                    
                                    <?php 
                                    $questions2 = [
                                        'fasilitator_persiapan' => 'Fasilitator menyiapkan segala sesuatunya dengan baik',
                                        'fasilitator_materi' => 'Fasilitator menguasai materi dan menyampaikan sesuai agenda',
                                        'waktu_training' => 'Waktu training sudah cukup dalam penyampaian materi',
                                        'training_interaktif' => 'Training dilakukan secara interaktif dan menarik',
                                        'fasilitas_tempat' => 'Fasilitas tempat dan konsumsi mendukung kenyamanan',
                                        'peralatan_training' => 'Peralatan training yang disediakan sudah memadai'
                                    ];
                                    
                                    $num = 1;
                                    foreach ($questions2 as $name => $question): ?>
                                        <div class="evaluation-question">
                                            <div class="question-text">
                                                <i class="bi bi-question-circle me-2"></i>
                                                <?= $num ?>. <?= $question ?> <span class="text-danger">*</span>
                                            </div>
                                            <div class="rating-scale">
                                                <span class="scale-label">Tidak Setuju</span>
                                                <div class="rating-options">
                                                    <?php for ($i = 1; $i <= 4; $i++): ?>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="<?= $name ?>" value="<?= $i ?>" 
                                                                   <?= (($_SESSION['eval_data'][$name] ?? 0) == $i) ? 'checked' : '' ?> required>
                                                            <label class="form-check-label"><?= $i ?></label>
                                                        </div>
                                                    <?php endfor; ?>
                                                </div>
                                                <span class="scale-label">Setuju</span>
                                            </div>
                                        </div>
                                        <?php $num++; ?>
                                    <?php endforeach; ?>

                                <?php elseif ($step === 3): ?>
                                    <!-- Step 3: Benefits & Relevance -->
                                    <div class="alert alert-info mb-4">
                                        <i class="bi bi-info-circle me-2"></i>
                                        Berikan penilaian dengan skala 1-4 (1=Sangat Tidak Setuju, 4=Sangat Setuju)
                                    </div>
                                    
                                    <?php 
                                    $questions3 = [
                                        'ilmu_baru' => 'Saya mendapatkan ilmu pengetahuan baru setelah mengikuti training ini',
                                        'memahami_tujuan' => 'Saya memahami tujuan dan materi training ini',
                                        'sesuai_harapan' => 'Program training ini sesuai dengan harapan saya',
                                        'bermanfaat' => 'Isi program training ini bermanfaat bagi diri saya dan pekerjaan',
                                        'kualitas_kerja' => 'Training ini bisa merubah kualitas pekerjaan saya menjadi lebih baik',
                                        'relevan_pekerjaan' => 'Training ini relevan dengan pekerjaan yang saya lakukan saat ini'
                                    ];
                                    
                                    $num = 1;
                                    foreach ($questions3 as $name => $question): ?>
                                        <div class="evaluation-question">
                                            <div class="question-text">
                                                <i class="bi bi-lightbulb me-2"></i>
                                                <?= $num ?>. <?= $question ?> <span class="text-danger">*</span>
                                            </div>
                                            <div class="rating-scale">
                                                <span class="scale-label">Tidak Setuju</span>
                                                <div class="rating-options">
                                                    <?php for ($i = 1; $i <= 4; $i++): ?>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="radio" name="<?= $name ?>" value="<?= $i ?>" 
                                                                   <?= (($_SESSION['eval_data'][$name] ?? 0) == $i) ? 'checked' : '' ?> required>
                                                            <label class="form-check-label"><?= $i ?></label>
                                                        </div>
                                                    <?php endfor; ?>
                                                </div>
                                                <span class="scale-label">Setuju</span>
                                            </div>
                                        </div>
                                        <?php $num++; ?>
                                    <?php endforeach; ?>

                                <?php elseif ($step === 4): ?>
                                    <!-- Step 4: Suggestions -->
                                    <div class="mb-4">
                                        <label class="form-label">
                                            <i class="bi bi-chat-text me-2"></i>Saran dan Masukan
                                        </label>
                                        <textarea class="form-control" name="saran_masukan" rows="6" 
                                                  placeholder="Tuliskan saran dan masukan Anda untuk perbaikan training selanjutnya..."><?= htmlspecialchars($_SESSION['eval_data']['saran_masukan'] ?? '') ?></textarea>
                                        <div class="form-text">
                                            <i class="bi bi-info-circle me-1"></i>
                                            Opsional - Berikan saran untuk meningkatkan kualitas training
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <!-- Navigation -->
                                <div class="d-flex justify-content-between mt-4">
                                    <div>
                                        <?php if ($step > 1): ?>
                                            <button type="submit" name="action" value="back" class="btn btn-secondary">
                                                <i class="bi bi-arrow-left me-2"></i>Kembali
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                    <div class="d-flex gap-2">
                                        <?php if ($step < 4): ?>
                                            <button type="submit" name="action" value="next" class="btn btn-primary">
                                                Selanjutnya<i class="bi bi-arrow-right ms-2"></i>
                                            </button>
                                        <?php else: ?>
                                            <button type="submit" name="action" value="submit" class="btn btn-primary">
                                                <i class="bi bi-check-circle me-2"></i>Kirim Evaluasi
                                            </button>
                                        <?php endif; ?>
                                        <a href="list_training.php" class="btn btn-outline-secondary">
                                            <i class="bi bi-x-circle me-2"></i>Batal
                                        </a>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <footer class="text-center">
        <div class="container">
            <p class="mb-0"><i class="bi bi-c-circle me-2"></i><?php echo date('Y'); ?> E-Learning System by PIM</p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebarToggle');
            const sidebar = document.getElementById('sidebar');
            const sidebarOverlay = document.getElementById('sidebarOverlay');
            const trainingDropdownToggle = document.getElementById('trainingDropdown');
            const postTestDropdownToggle = document.getElementById('postTestDropdown');
            
            let isDropdownClick = false;

            function toggleSidebar() {
                sidebar.classList.toggle('sidebar-active');
                sidebarOverlay.style.display = sidebar.classList.contains('sidebar-active') ? 'block' : 'none';
                
                if (sidebar.classList.contains('sidebar-active')) {
                    document.body.style.overflow = 'hidden';
                } else {
                    document.body.style.overflow = '';
                }
            }

            sidebarToggle.addEventListener('click', function(e) {
                e.stopPropagation();
                toggleSidebar();
            });

            sidebarOverlay.addEventListener('click', function() {
                if (!isDropdownClick) {
                    toggleSidebar();
                }
                isDropdownClick = false;
            });

            if (trainingDropdownToggle) {
                trainingDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }

            if (postTestDropdownToggle) {
                postTestDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }
            
            document.addEventListener('click', function(e) {
                if (window.innerWidth <= 768) {
                    let clickedInsideAnyDropdown = false;
                    if (trainingDropdownToggle && trainingDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (postTestDropdownToggle && postTestDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (!clickedInsideAnyDropdown) {
                        isDropdownClick = false;
                    }
                }
            });

            const regularNavLinks = document.querySelectorAll('.sidebar .nav-link:not(.dropdown-toggle)');
            regularNavLinks.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            const dropdownItems = document.querySelectorAll('.sidebar .dropdown-menu .dropdown-item');
            dropdownItems.forEach(item => {
                item.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            window.addEventListener('resize', function() {
                if (window.innerWidth > 768) {
                    sidebar.classList.remove('sidebar-active');
                    sidebarOverlay.style.display = 'none';
                    document.body.style.overflow = '';
                    isDropdownClick = false;
                }
            });

            // Auto format date input
            const dateInput = document.querySelector('input[name="training_date"]');
            if (dateInput) {
                dateInput.addEventListener('input', function(e) {
                    let value = e.target.value.replace(/\D/g, '');
                    if (value.length >= 2) value = value.substring(0, 2) + '-' + value.substring(2);
                    if (value.length >= 5) value = value.substring(0, 5) + '-' + value.substring(5, 9);
                    e.target.value = value;
                });
            }
        });
    </script>
</body>
</html>