<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once 'koneksi.php';

// Cek login
checkLogin();

date_default_timezone_set('Asia/Jakarta');

$user_id = $_SESSION['user_id'];
$role = $_SESSION['role'];
$nik_karyawan = $_SESSION['nik_karyawan'];

// Pastikan user adalah trainer atau admin
if (!isset($role) || ($role !== 'trainer' && $role !== 'admin')) {
    header("Location: login.php?error=Anda tidak memiliki izin untuk mengakses halaman ini");
    exit();
}

// Ambil informasi user menggunakan prepared statement
if ($role === 'trainer') {
    $stmt = $koneksi->prepare("SELECT nama_lengkap, jabatan, divisi, foto_profil FROM trainers WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
} else {
    // Untuk admin, ambil dari tabel yang sesuai
    $stmt = $koneksi->prepare("SELECT nama_lengkap, jabatan, divisi, foto_profil FROM admins WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
}

// Ambil input pencarian
$search_term = isset($_GET['search']) ? trim($_GET['search']) : '';
$error = isset($_GET['error']) ? $_GET['error'] : '';
$evaluation_data = [];
$search_performed = false;

if (!empty($search_term)) {
    $search_performed = true;
    // Query untuk mencari evaluasi berdasarkan nama training
    $stmt = $koneksi->prepare("
        SELECT 
            id,
            training_name,
            instructor,
            location,
            participant_name,
            position,
            work_unit,
            training_date,
            fasilitator_persiapan,
            fasilitator_materi,
            waktu_training,
            training_interaktif,
            fasilitas_tempat,
            peralatan_training,
            ilmu_baru,
            memahami_tujuan,
            sesuai_harapan,
            bermanfaat,
            kualitas_kerja,
            relevan_pekerjaan,
            saran_masukan,
            created_at
        FROM 
            evaluasi_training
        WHERE 
            training_name LIKE ?
        ORDER BY 
            created_at DESC
    ");
    $search_param = "%" . $search_term . "%";
    $stmt->bind_param("s", $search_param);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $evaluation_data[] = $row;
    }
    $stmt->close();
}

$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Data Evaluasi Training - E-Learning</title>
    <link rel="icon" type="image/png" href="image/icon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        :root {
            --soft-ui-primary-start: #00C4CC; /* Teal */
            --soft-ui-primary-end: #CB0C9F; /* Purple */
            --soft-ui-secondary: #67748E; /* Secondary color */
            --soft-ui-info: #17C1E8; /* Info color */
            --soft-ui-success: #82D616; /* Success color */
            --soft-ui-warning: #FB8C00; /* Warning color */
            --soft-ui-danger: #EA0606; /* Danger color */
            --soft-ui-light: #E9ECEF; /* Light color */
            --soft-ui-dark: #344767; /* Dark color */
            --soft-ui-white: #FFFFFF;
            --soft-ui-gray-100: #F8F9FA;
            --soft-ui-gray-200: #E9ECEF;
            --soft-ui-gray-300: #DEE2E6;
            --soft-ui-gray-400: #CED4DA;
            --soft-ui-gray-500: #ADB5BD;
            --soft-ui-gray-600: #6C757D;
            --soft-ui-gray-700: #495057;
            --soft-ui-gray-800: #343A40;
            --soft-ui-gray-900: #212529;
        }

        html, body {
            height: 100%;
            margin: 0;
            font-family: 'Open Sans', sans-serif; /* Soft UI font */
            background-color: var(--soft-ui-gray-100); /* Light background */
        }
        
        body {
            overflow-x: hidden;
            display: flex;
            flex-direction: column;
        }
        
        .navbar-top {
            position: fixed;
            top: 0;
            width: 100%;
            z-index: 1100;
            background: var(--soft-ui-white);
            border-bottom: 1px solid var(--soft-ui-gray-200);
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.05);
            padding: 0.75rem 1.5rem;
        }
        
        .navbar-top .container-fluid {
            display: flex;
            align-items: center;
            justify-content: space-between;
            flex-wrap: nowrap;
            position: relative;
        }
        
        /* Logo styling */
        .navbar-logo {
            height: 40px;
            width: auto;
            margin-right: 15px;
        }
        
        /* Center title styling */
        .navbar-title {
            position: absolute;
            left: 50%;
            transform: translateX(-50%);
            font-size: 1.3rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            white-space: nowrap;
            pointer-events: none; /* Prevent interference with dropdown */
        }

        /* Mobile title styling - 2 lines */
        .navbar-title-mobile {
            display: none;
            font-size: 0.85rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            line-height: 1.2;
            flex: 1;
            margin: 0 10px;
        }

        .navbar-title-mobile .line1 {
            display: block;
        }

        .navbar-title-mobile .line2 {
            display: block;
            font-size: 0.8rem;
        }
        
        .navbar-brand {
            font-size: 1.2rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            display: flex;
            align-items: center;
            z-index: 1;
        }

        .navbar-nav {
            position: relative;
            z-index: 1200; /* Higher z-index for dropdown */
        }

        .navbar-nav .nav-item .nav-link {
            color: var(--soft-ui-secondary);
            font-weight: 600;
            padding: 0.5rem 1rem;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .nav-item .nav-link:hover {
            color: var(--soft-ui-primary-end);
        }

        .navbar-nav .nav-item .nav-link.active {
            color: var(--soft-ui-primary-end);
        }

        /* Fix dropdown menu positioning and z-index */
        .navbar-nav .dropdown-menu {
            position: absolute;
            top: 100%;
            right: 0;
            left: auto;
            z-index: 1300; /* Very high z-index */
            background-color: var(--soft-ui-white);
            border: 1px solid var(--soft-ui-gray-200);
            border-radius: 0.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            min-width: 160px;
            margin-top: 0.125rem;
        }

        .navbar-nav .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .dropdown-item:hover {
            background-color: var(--soft-ui-gray-200);
            color: var(--soft-ui-dark);
        }

        .navbar-nav .dropdown-item.active {
            background-color: var(--soft-ui-primary-end);
            color: var(--soft-ui-white);
        }
        
        .sidebar {
            min-height: 100vh;
            width: 250px;
            position: fixed;
            top: 0;
            left: 0;
            z-index: 1000;
            background: var(--soft-ui-white);
            color: var(--soft-ui-dark);
            padding-top: 70px; /* Space for top navbar */
            transition: all 0.3s ease-in-out;
            border-right: 1px solid var(--soft-ui-gray-200);
            box-shadow: 4px 0 15px rgba(0, 0, 0, 0.05);
            border-radius: 0 1rem 1rem 0; /* Rounded right corners */
        }
        
        .content {
            margin-left: 250px;
            margin-top: 0;
            padding: 20px;
            padding-top: 90px; /* Adjust for fixed top navbar */
            transition: all 0.3s ease-in-out;
            flex-grow: 1;
        }
        
        .hamburger-mobile {
            display: none;
            background: none;
            border: none;
            color: var(--soft-ui-dark);
            font-size: 1.5rem;
            cursor: pointer;
            margin-right: 15px;
            transition: all 0.3s ease;
            z-index: 1;
        }
        
        .hamburger-mobile:focus {
            outline: none;
        }
        
        .sidebar-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(0, 0, 0, 0.4);
            z-index: 999;
        }
        
        /* Profile Section Styles (adapted for Soft UI) */
        .profile-section {
            text-align: center;
            padding: 15px 20px;
            border-bottom: 1px solid var(--soft-ui-gray-200);
            background: var(--soft-ui-white);
            border-radius: 0.75rem;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .profile-photo {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--soft-ui-gray-300);
            margin: 0 auto 10px;
            display: block;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: var(--soft-ui-gray-400);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            border: 2px solid var(--soft-ui-gray-300);
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder i {
            font-size: 2rem;
            color: var(--soft-ui-white);
        }
        
        .profile-name {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--soft-ui-dark);
            margin-bottom: 5px;
            word-wrap: break-word;
        }
        
        .profile-info {
            font-size: 0.9rem;
            font-weight: 400;
            color: var(--soft-ui-secondary);
            margin-bottom: 3px;
        }
        
        /* Sidebar Menu Styles */
        .sidebar .nav-link {
            color: var(--soft-ui-secondary);
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
            transition: all 0.2s ease-in-out;
            font-weight: 600;
            display: flex;
            align-items: center;
        }
        
        .sidebar .nav-link:hover {
            background: linear-gradient(90deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-dark);
        }
        
        .sidebar .nav-link.active {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white) !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .sidebar .nav-link.active i {
            color: var(--soft-ui-white);
        }
        
        .sidebar .nav-link i {
            font-size: 1.2rem;
            margin-right: 10px;
            color: var(--soft-ui-secondary);
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-menu {
            background-color: var(--soft-ui-white);
            border: none;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            border-radius: 0.5rem;
            padding: 0.5rem 0;
            margin-top: 0.25rem;
        }

        .sidebar .nav-item.dropdown .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-item:hover {
            background: linear-gradient(90deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-dark);
        }

        .sidebar .nav-item.dropdown .dropdown-item.active {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
        }

        .sidebar .nav-item:not(:last-child) {
            border-bottom: none; /* Remove default border */
        }
        
        /* Card Styles - Updated to Soft UI */
        .card {
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            border: 1px solid var(--soft-ui-gray-200);
            border-radius: 0.75rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            overflow: hidden;
            background: var(--soft-ui-white);
            backdrop-filter: blur(10px);
        }
        
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.08);
        }
        
        .card-header {
            padding: 25px 30px;
            color: var(--soft-ui-white);
            text-align: center;
            font-weight: 700;
            font-size: 1.4rem;
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            position: relative;
            border-radius: 0.75rem 0.75rem 0 0;
        }

        .card-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(45deg, rgba(255, 255, 255, 0.1) 0%, transparent 50%);
            pointer-events: none;
        }

        .card-header i {
            margin-right: 10px;
            font-size: 1.3rem;
        }
        
        .card-body {
            padding: 30px;
            background: var(--soft-ui-white);
        }
        
        /* Table Styles - Updated to Soft UI */
        .table-responsive {
            margin-top: 20px;
            border-radius: 0.75rem;
            overflow: hidden;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        }
        
        .table {
            margin-bottom: 0;
        }

        .table th, .table td {
            vertical-align: middle;
            text-align: center;
            font-size: 0.95rem;
            padding: 15px 12px;
            border: none;
        }
        
        .table thead th {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            font-size: 0.85rem;
        }

        .table tbody tr {
            background: var(--soft-ui-white);
            transition: all 0.2s ease-in-out;
        }
        
        .table tbody tr:hover {
            background: linear-gradient(195deg, rgba(0, 196, 204, 0.05), rgba(203, 12, 159, 0.05));
            transform: scale(1.01);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
        }

        .table tbody tr:nth-child(even) {
            background: var(--soft-ui-gray-100);
        }

        .table tbody tr:nth-child(even):hover {
            background: linear-gradient(195deg, rgba(0, 196, 204, 0.08), rgba(203, 12, 159, 0.08));
        }
        
        /* Rata kiri untuk kolom Nama Training */
        .table td:first-child {
            text-align: left !important;
            font-weight: 600;
            color: var(--soft-ui-dark);
        }
        
        .rating-display {
            display: inline-flex;
            gap: 2px;
            margin-top: 4px;
        }
        
        .rating-star {
            color: #ffc107;
            font-size: 0.9rem;
        }
        
        .rating-star.empty {
            color: var(--soft-ui-gray-300);
        }
        
        .rating-column {
            min-width: 140px;
        }
        
        /* Export buttons styling - Updated to Soft UI */
        .export-buttons {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .btn-export {
            padding: 0.75rem 1rem;
            border-radius: 0.75rem;
            border: none;
            font-size: 1.2rem;
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            font-weight: 600;
        }
        
        .btn-export:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.15);
        }
        
        .btn-excel {
            background: linear-gradient(195deg, var(--soft-ui-success), #20c997);
            color: var(--soft-ui-white);
        }
        
        .btn-excel:hover {
            background: linear-gradient(195deg, #20c997, var(--soft-ui-success));
            color: var(--soft-ui-white);
        }
        
        .btn-pdf {
            background: linear-gradient(195deg, var(--soft-ui-danger), #dc3545);
            color: var(--soft-ui-white);
        }
        
        .btn-pdf:hover {
            background: linear-gradient(195deg, #dc3545, var(--soft-ui-danger));
            color: var(--soft-ui-white);
        }
        
        /* Button Styles - Updated to Soft UI */
        .btn-primary {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            border: none;
            border-radius: 0.75rem;
            padding: 0.75rem 2rem;
            font-weight: 600;
            font-size: 1rem;
            box-shadow: 0 4px 12px rgba(0, 196, 204, 0.3);
            transition: all 0.3s ease;
            color: var(--soft-ui-white);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .btn-primary:hover {
            background: linear-gradient(195deg, #009ea6, #a80a85);
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 196, 204, 0.4);
            color: var(--soft-ui-white);
        }

        .btn-info {
            background: linear-gradient(195deg, var(--soft-ui-info), #0ea5e9);
            border: none;
            border-radius: 0.5rem;
            color: var(--soft-ui-white);
            font-weight: 600;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(23, 193, 232, 0.3);
            padding: 0.4rem 0.8rem;
            font-size: 0.85rem;
        }

        .btn-info:hover {
            background: linear-gradient(195deg, #0ea5e9, var(--soft-ui-info));
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(23, 193, 232, 0.4);
            color: var(--soft-ui-white);
        }

        /* Badge Styles - Updated to Soft UI */
        .badge {
            font-size: 0.8rem;
            padding: 0.4rem 0.8rem;
            border-radius: 0.5rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.3px;
        }

        .bg-success {
            background: linear-gradient(195deg, var(--soft-ui-success), #20c997) !important;
            color: var(--soft-ui-white) !important;
        }

        .bg-info {
            background: linear-gradient(195deg, var(--soft-ui-info), #0ea5e9) !important;
            color: var(--soft-ui-white) !important;
        }
        
        /* Alert Styles - Updated to Soft UI */
        .alert {
            border-radius: 0.75rem;
            padding: 1.5rem;
            border: none;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            font-weight: 500;
        }

        .alert-danger {
            background: linear-gradient(195deg, rgba(234, 6, 6, 0.1), rgba(220, 53, 69, 0.1));
            color: var(--soft-ui-danger);
            border-left: 4px solid var(--soft-ui-danger);
        }

        .alert-warning {
            background: linear-gradient(195deg, rgba(251, 140, 0, 0.1), rgba(255, 193, 7, 0.1));
            color: var(--soft-ui-warning);
            border-left: 4px solid var(--soft-ui-warning);
        }

        .alert-info {
            background: linear-gradient(195deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-primary-start);
            border-left: 4px solid var(--soft-ui-primary-start);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 700;
            color: var(--soft-ui-dark);
            margin-bottom: 10px;
            text-align: left;
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .page-subtitle {
            font-size: 1.2rem;
            font-weight: 500;
            color: var(--soft-ui-secondary);
            margin-bottom: 30px;
            text-align: left;
        }
        
        /* Search Form Styles - Updated to Soft UI */
        .search-form {
            margin-bottom: 25px;
        }
        
        .search-form .input-group {
            flex-direction: row;
            align-items: stretch;
        }
        
        .search-form .form-control {
            border-radius: 0.75rem;
            padding: 12px 16px;
            border: 1px solid var(--soft-ui-gray-300);
            font-size: 1rem;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
        }

        .search-form .form-control:focus {
            border-color: var(--soft-ui-primary-start);
            box-shadow: 0 4px 12px rgba(0, 196, 204, 0.2);
            outline: none;
        }
        
        .search-form .btn {
            margin-left: 15px;
            border-radius: 0.75rem;
        }
        
        footer {
            background: var(--soft-ui-white);
            color: var(--soft-ui-secondary);
            border-top: 1px solid var(--soft-ui-gray-200);
            padding: 1rem 0;
            text-align: center;
            flex-shrink: 0;
            margin-top: 2rem;
        }
        
        .animate-fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        /* Modal Styles - Updated to Soft UI */
        .modal-content {
            border-radius: 0.75rem;
            border: none;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        .modal-header {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
            border-radius: 0.75rem 0.75rem 0 0;
            border-bottom: none;
            padding: 20px 25px;
        }

        .modal-title {
            font-weight: 700;
            font-size: 1.3rem;
        }

        .btn-close {
            filter: brightness(0) invert(1);
        }

        .modal-body {
            padding: 25px;
        }

        .modal-footer {
            border-top: 1px solid var(--soft-ui-gray-200);
            padding: 20px 25px;
        }

        .btn-secondary {
            background: var(--soft-ui-gray-500);
            border: none;
            border-radius: 0.5rem;
            color: var(--soft-ui-white);
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-secondary:hover {
            background: var(--soft-ui-gray-600);
            color: var(--soft-ui-white);
            transform: translateY(-1px);
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            .sidebar {
                margin-left: -250px;
                border-radius: 0;
                padding-top: 90px; /* Increased for mobile navbar height */
            }
            .sidebar-active {
                margin-left: 0;
            }
            .content {
                margin-left: 0;
                padding-top: 100px; /* Increased for mobile navbar height */
            }
            
            /* Hide desktop title and logo on mobile */
            .navbar-title {
                display: none;
            }
            .navbar-logo {
                display: none;
            }
            
            /* Show mobile title */
            .navbar-title-mobile {
                display: block;
            }
            
            .hamburger-mobile {
                display: block;
            }
            .navbar-search {
                display: none; /* Hide search on small screens */
            }
            .navbar-nav {
                position: static;
            }
            .navbar-nav .dropdown-menu {
                position: absolute;
                right: 0;
                left: auto;
                z-index: 1300;
            }
            
            /* Adjust navbar height for mobile */
            .navbar-top {
                padding: 0.5rem 1rem;
                min-height: 70px;
            }

            .profile-photo, .profile-photo-placeholder {
                width: 60px;
                height: 60px;
            }
            .profile-photo-placeholder i {
                font-size: 1.5rem;
            }
            .profile-name {
                font-size: 1rem;
            }
            .profile-info {
                font-size: 0.8rem;
            }

            .page-title { 
                font-size: 1.5rem; 
                text-align: left;
            }
            .page-subtitle { 
                font-size: 1rem; 
                margin-top: -5px; 
            }
            .btn-primary { 
                padding: 0.6rem 1.5rem; 
                font-size: 0.9rem; 
            }
            .alert { 
                font-size: 0.9rem; 
            }
            .search-form .input-group { 
                flex-direction: column; 
                align-items: stretch; 
            }
            .search-form .form-control { 
                margin-bottom: 15px; 
                width: 100%; 
            }
            .search-form .btn { 
                width: 100%; 
                margin-left: 0;
            }
            .table th, .table td { 
                font-size: 0.85rem; 
                padding: 10px 8px; 
            }
            .rating-column { 
                min-width: 120px; 
            }
            .export-buttons { 
                flex-direction: column; 
                gap: 10px; 
                width: 100%;
                margin-bottom: 15px;
            }
            .btn-export { 
                padding: 0.6rem 1rem; 
                font-size: 1rem;
                width: 100%;
            }
        }
        
        @media (max-width: 576px) {
            .navbar-top {
                padding: 0.5rem 1rem;
                min-height: 70px;
            }
            .navbar-title-mobile {
                font-size: 0.8rem;
            }
            .navbar-title-mobile .line2 {
                font-size: 0.75rem;
            }
            
            .profile-photo, .profile-photo-placeholder {
                width: 50px;
                height: 50px;
            }
            .profile-photo-placeholder i {
                font-size: 1.2rem;
            }
            .profile-name {
                font-size: 0.9rem;
            }
            .profile-info {
                font-size: 0.7rem;
            }

            .page-title { 
                font-size: 1.3rem; 
            }
            .page-subtitle { 
                font-size: 0.9rem; 
                margin-top: -5px; 
            }
            .btn-primary { 
                padding: 0.5rem 1.2rem; 
                font-size: 0.85rem; 
            }
            .alert { 
                font-size: 0.85rem; 
            }
            .table th, .table td { 
                font-size: 0.8rem; 
                padding: 8px 6px; 
            }
            .rating-column { 
                min-width: 100px; 
            }
            .btn-export { 
                padding: 0.5rem 0.8rem; 
                font-size: 0.9rem;
            }
        }
    </style>
</head>
<body>
    <!-- Top Navbar -->
    <nav class="navbar navbar-expand-lg navbar-top">
        <div class="container-fluid">
            <button class="hamburger-mobile d-md-none" type="button" id="sidebarToggle">
                <i class="bi bi-list"></i>
            </button>
            <a class="navbar-brand d-none d-md-flex" href="#">
                <img src="image/logopim.png" alt="Logo PIM" class="navbar-logo">
            </a>
            <div class="navbar-title d-none d-md-block">
                E-Learning System Pondok Indah Mall
            </div>
            <div class="navbar-title-mobile d-md-none">
                <span class="line1">E-Learning System</span>
                <span class="line2">Pondok Indah Mall</span>
            </div>
            <ul class="navbar-nav ms-auto mb-2 mb-lg-0">
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle text-dark" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-person-circle"></i>
                    </a>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><a class="dropdown-item <?php echo ($current_page == 'profile.php') ? 'active' : ''; ?>" href="profile.php">Profile</a></li>
                        <li><a class="dropdown-item <?php echo ($current_page == 'change_password.php') ? 'active' : ''; ?>" href="change_password.php">Change Password</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="logout.php">Logout</a></li>
                    </ul>
                </li>
            </ul>
        </div>
    </nav>

    <!-- Sidebar overlay (for mobile) -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Sidebar -->
    <div class="sidebar border-end" id="sidebar">
        <!-- Profile Section moved here, directly after sidebar padding-top -->
        <div class="profile-section">
            <?php if (!empty($user['foto_profil'])): ?>
                <img src="Uploads/profiles/<?php echo htmlspecialchars($user['foto_profil']); ?>" alt="Foto Profil" class="profile-photo">
            <?php else: ?>
                <div class="profile-photo-placeholder">
                    <i class="bi bi-person-circle"></i>
                </div>
            <?php endif; ?>
            <div class="profile-name"><?php echo htmlspecialchars($user['nama_lengkap'] ?? 'User Tidak Dikenal'); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($user['jabatan'] ?? 'Tidak ada jabatan'); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($user['divisi'] ?? 'Tidak ada divisi'); ?></div>
        </div>
        <div class="p-3">
            <h5 class="mb-4 text-dark">Menu</h5>
            <ul class="nav flex-column">
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'dashboard.php') ? 'active' : ''; ?>" href="dashboard.php">
                        <i class="bi bi-house-door-fill"></i> Dashboard
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'materi.php') ? 'active' : ''; ?>" href="materi.php">
                        <i class="bi bi-book"></i> Materi
                    </a>
                </li>
                <!-- Post Test Menu -->
                <?php if ($role == 'trainer'): ?>
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_posttest.php' || $current_page == 'list_posttest.php' || $current_page == 'view_posttest_scores.php') ? 'active' : ''; ?>" href="#" id="postTestDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="bi bi-clipboard-check"></i> Post Test
                        </a>
                        <ul class="dropdown-menu" aria-labelledby="postTestDropdown">
                            <li><a class="dropdown-item <?php echo ($current_page == 'add_posttest.php') ? 'active' : ''; ?>" href="add_posttest.php">Tambah Post Test</a></li>
                            <li><a class="dropdown-item <?php echo ($current_page == 'list_posttest.php' || $current_page == 'view_posttest_scores.php') ? 'active' : ''; ?>" href="list_posttest.php">List Post Test</a></li>
                        </ul>
                    </li>
                <?php elseif ($role == 'peserta' || $role == 'iss'): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'list_posttest.php' || $current_page == 'take_posttest.php' || $current_page == 'post_test_result.php') ? 'active' : ''; ?>" href="list_posttest.php">
                            <i class="bi bi-clipboard-check"></i> Post Test
                        </a>
                    </li>
                <?php endif; ?>
                <!-- End Post Test Menu -->
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_training.php' || $current_page == 'list_training.php') ? 'active' : ''; ?>" href="#" id="trainingDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-journal-text"></i> Training
                    </a>
                    <ul class="dropdown-menu" aria-labelledby="trainingDropdown">
                        <li><a class="dropdown-item <?php echo ($current_page == 'add_training.php') ? 'active' : ''; ?>" href="add_training.php">Tambah Kegiatan Training</a></li>
                        <li><a class="dropdown-item <?php echo ($current_page == 'list_training.php') ? 'active' : ''; ?>" href="list_training.php">List Training</a></li>
                    </ul>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'report.php') ? 'active' : ''; ?>" href="report.php">
                        <i class="bi bi-file-earmark-text"></i> Report
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'data_evaluasi.php') ? 'active' : ''; ?>" href="data_evaluasi.php">
                        <i class="bi bi-clipboard-data"></i> Data Evaluasi
                    </a>
                </li>
            </ul>
        </div>
    </div>

    <!-- Content -->
    <div class="content">
        <div class="container mt-4">
            <h2 class="page-title animate-fade-in">
                <i class="bi bi-clipboard-data me-3"></i>
                Data Evaluasi Training
            </h2>
            <p class="page-subtitle animate-fade-in">Pondok Indah Mall</p>
            <div class="row">
                <div class="col-md-12">
                    <div class="card animate-fade-in">
                        <div class="card-header">
                            <i class="bi bi-table"></i>
                            Data Evaluasi Training Karyawan
                        </div>
                        <div class="card-body">
                            <!-- Search Form -->
                            <form method="GET" action="" class="search-form">
                                <div class="input-group">
                                    <input type="text" class="form-control" name="search" placeholder="Cari nama training..." value="<?php echo htmlspecialchars($search_term); ?>">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-search me-2"></i> Cari
                                    </button>
                                </div>
                            </form>
                            
                            <?php if (!empty($error)): ?>
                                <div class="alert alert-danger mt-3">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <?php echo htmlspecialchars($error); ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($search_performed): ?>
                                <?php if (empty($evaluation_data)): ?>
                                    <div class="alert alert-warning mt-3">
                                        <i class="bi bi-info-circle me-2"></i>
                                        Tidak ditemukan data evaluasi untuk training dengan nama "<?php echo htmlspecialchars($search_term); ?>".
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-info mt-3">
                                        <i class="bi bi-search me-2"></i>
                                        Menampilkan hasil pencarian untuk "<?php echo htmlspecialchars($search_term); ?>".
                                    </div>
                                    <div class="table-responsive">
                                        <table class="table table-bordered table-striped">
                                            <thead>
                                                <tr>
                                                    <th><i class="bi bi-journal-text me-1"></i>Nama Training</th>
                                                    <th><i class="bi bi-person me-1"></i>Peserta</th>
                                                    <th><i class="bi bi-briefcase me-1"></i>Jabatan</th>
                                                    <th><i class="bi bi-building me-1"></i>Unit Kerja</th>
                                                    <th><i class="bi bi-calendar me-1"></i>Tanggal Training</th>
                                                    <th><i class="bi bi-person-gear me-1"></i>Instruktur</th>
                                                    <th><i class="bi bi-geo-alt me-1"></i>Lokasi</th>
                                                    <th class="rating-column"><i class="bi bi-emoji-smile me-1"></i>Rata-rata Reaksi</th>
                                                    <th class="rating-column"><i class="bi bi-mortarboard me-1"></i>Rata-rata Pembelajaran</th>
                                                    <th><i class="bi bi-gear me-1"></i>Aksi</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($evaluation_data as $data): ?>
                                                    <?php
                                                    // Hitung rata-rata Reaksi (field 1-6)
                                                    $reaction_ratings = [
                                                        $data['fasilitator_persiapan'],    // 1
                                                        $data['fasilitator_materi'],       // 2
                                                        $data['waktu_training'],           // 3
                                                        $data['training_interaktif'],      // 4
                                                        $data['fasilitas_tempat'],         // 5
                                                        $data['peralatan_training']        // 6
                                                    ];
                                                    $reaction_average = array_sum($reaction_ratings) / count($reaction_ratings);
                                                    $reaction_average_rounded = round($reaction_average, 1);
                                                    
                                                    // Hitung rata-rata Pembelajaran (field 7-12)
                                                    $learning_ratings = [
                                                        $data['ilmu_baru'],                // 7
                                                        $data['memahami_tujuan'],          // 8
                                                        $data['sesuai_harapan'],           // 9
                                                        $data['bermanfaat'],               // 10
                                                        $data['kualitas_kerja'],           // 11
                                                        $data['relevan_pekerjaan']         // 12
                                                    ];
                                                    $learning_average = array_sum($learning_ratings) / count($learning_ratings);
                                                    $learning_average_rounded = round($learning_average, 1);
                                                    
                                                    // Format tanggal
                                                    $training_date = date('d-m-Y', strtotime($data['training_date']));
                                                    ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($data['training_name']); ?></td>
                                                        <td><strong><?php echo htmlspecialchars($data['participant_name']); ?></strong></td>
                                                        <td><?php echo htmlspecialchars($data['position']); ?></td>
                                                        <td><?php echo htmlspecialchars($data['work_unit']); ?></td>
                                                        <td><small class="text-muted"><?php echo $training_date; ?></small></td>
                                                        <td><?php echo htmlspecialchars($data['instructor']); ?></td>
                                                        <td><?php echo htmlspecialchars($data['location']); ?></td>
                                                        <td class="rating-column">
                                                            <span class="badge bg-success"><?php echo $reaction_average_rounded; ?>/4</span>
                                                            <div class="rating-display">
                                                                <?php for ($i = 1; $i <= 4; $i++): ?>
                                                                    <i class="bi bi-star-fill rating-star <?php echo $i <= $reaction_average_rounded ? '' : 'empty'; ?>"></i>
                                                                <?php endfor; ?>
                                                            </div>
                                                        </td>
                                                        <td class="rating-column">
                                                            <span class="badge bg-info"><?php echo $learning_average_rounded; ?>/4</span>
                                                            <div class="rating-display">
                                                                <?php for ($i = 1; $i <= 4; $i++): ?>
                                                                    <i class="bi bi-star-fill rating-star <?php echo $i <= $learning_average_rounded ? '' : 'empty'; ?>"></i>
                                                                <?php endfor; ?>
                                                            </div>
                                                        </td>
                                                        <td>
                                                            <button class="btn btn-sm btn-info" onclick="viewDetail(<?php echo $data['id']; ?>)">
                                                                <i class="bi bi-eye me-1"></i> Detail
                                                            </button>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            <?php elseif (!empty($search_term)): ?>
                                <div class="alert alert-info mt-3">
                                    <i class="bi bi-search me-2"></i>
                                    Pencarian untuk "<?php echo htmlspecialchars($search_term); ?>" telah dilakukan.
                                </div>
                            <?php endif; ?>
                            
                            <div class="d-flex justify-content-between align-items-center mt-4">
                                <!-- Export Buttons - Hanya muncul jika ada data hasil pencarian -->
                                <?php if ($search_performed && !empty($evaluation_data)): ?>
                                    <div class="export-buttons">
                                        <button class="btn btn-export btn-excel" onclick="exportToExcel()" title="Export ke Excel">
                                            <i class="bi bi-file-earmark-excel me-2"></i>Excel
                                        </button>
                                        <button class="btn btn-export btn-pdf" onclick="exportToPDF()" title="Export ke PDF">
                                            <i class="bi bi-file-earmark-pdf me-2"></i>PDF
                                        </button>
                                    </div>
                                <?php else: ?>
                                    <div></div> <!-- Empty div untuk spacing -->
                                <?php endif; ?>
                                
                                <!-- Back Button -->
                                <a href="dashboard.php" class="btn btn-primary">
                                    <i class="bi bi-arrow-left me-2"></i> Kembali
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Detail Evaluasi -->
    <div class="modal fade" id="detailModal" tabindex="-1" aria-labelledby="detailModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="detailModalLabel">
                        <i class="bi bi-info-circle me-2"></i>
                        Detail Evaluasi Training
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="modalContent">
                    <!-- Content will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle me-1"></i>
                        Tutup
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="text-center">
        <div class="container">
            <p class="mb-0">© <?php echo date('Y'); ?> E-Learning System by PIM</p>
        </div>
    </footer>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebarToggle');
            const sidebar = document.getElementById('sidebar');
            const sidebarOverlay = document.getElementById('sidebarOverlay');
            const trainingDropdownToggle = document.getElementById('trainingDropdown');
            const postTestDropdownToggle = document.getElementById('postTestDropdown');
            
            let isDropdownClick = false;

            function toggleSidebar() {
                sidebar.classList.toggle('sidebar-active');
                sidebarOverlay.style.display = sidebar.classList.contains('sidebar-active') ? 'block' : 'none';
                
                if (sidebar.classList.contains('sidebar-active')) {
                    document.body.style.overflow = 'hidden';
                } else {
                    document.body.style.overflow = '';
                }
            }

            sidebarToggle.addEventListener('click', function(e) {
                e.stopPropagation();
                toggleSidebar();
            });

            sidebarOverlay.addEventListener('click', function() {
                if (!isDropdownClick) {
                    toggleSidebar();
                }
                isDropdownClick = false;
            });

            if (trainingDropdownToggle) {
                trainingDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }

            if (postTestDropdownToggle) {
                postTestDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }
            
            document.addEventListener('click', function(e) {
                if (window.innerWidth <= 768) {
                    let clickedInsideAnyDropdown = false;
                    if (trainingDropdownToggle && trainingDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (postTestDropdownToggle && postTestDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (!clickedInsideAnyDropdown) {
                        isDropdownClick = false;
                    }
                }
            });

            const regularNavLinks = document.querySelectorAll('.sidebar .nav-link:not(.dropdown-toggle)');
            regularNavLinks.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            const dropdownItems = document.querySelectorAll('.sidebar .dropdown-menu .dropdown-item');
            dropdownItems.forEach(item => {
                item.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            window.addEventListener('resize', function() {
                if (window.innerWidth > 768) {
                    sidebar.classList.remove('sidebar-active');
                    sidebarOverlay.style.display = 'none';
                    document.body.style.overflow = '';
                    isDropdownClick = false;
                }
            });
        });

        function viewDetail(evaluationId) {
            // AJAX call to get evaluation details
            fetch('get_evaluation_detail.php?id=' + evaluationId)
                .then(response => response.text())
                .then(data => {
                    document.getElementById('modalContent').innerHTML = data;
                    new bootstrap.Modal(document.getElementById('detailModal')).show();
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Terjadi kesalahan saat memuat detail evaluasi');
                });
        }

        // Export functions
        function exportToExcel() {
            const searchTerm = '<?php echo addslashes($search_term); ?>';
            if (searchTerm) {
                window.open('export_evaluasi_excel.php?search=' + encodeURIComponent(searchTerm), '_blank');
            } else {
                alert('Silakan lakukan pencarian terlebih dahulu');
            }
        }

        function exportToPDF() {
            const searchTerm = '<?php echo addslashes($search_term); ?>';
            if (searchTerm) {
                window.open('export_evaluasi_pdf.php?search=' + encodeURIComponent(searchTerm), '_blank');
            } else {
                alert('Silakan lakukan pencarian terlebih dahulu');
            }
        }

        // Gunakan event delegation untuk menangkap klik pada button PDF
        document.addEventListener('click', function(event) {
            // Cek apakah yang diklik adalah button PDF
            if (event.target.closest('.btn-export-pdf')) {
                // Ambil ID evaluasi dari data attribute
                const evaluationId = event.target.closest('.btn-export-pdf').dataset.id;
                if (evaluationId) {
                    console.log('Downloading PDF for ID:', evaluationId);
                    window.open('export_evaluation_pdf.php?id=' + evaluationId, '_blank');
                }
            }
        });
    </script>
</body>
</html>