<?php

require_once 'koneksi.php';

// Cek login
checkLogin();
// Cek apakah user adalah trainer
checkTrainer();

$user_id = $_SESSION['user_id'];
$nik_karyawan = $_SESSION['nik_karyawan'];
$error = "";
$success = "";

// Ambil informasi trainer berdasarkan nik_karyawan
$stmt = $koneksi->prepare("SELECT nama_lengkap, foto_profil, jabatan, divisi FROM trainers WHERE nik_karyawan = ?");
$stmt->bind_param("s", $nik_karyawan);
$stmt->execute();
$result = $stmt->get_result();
$trainer = $result->fetch_assoc();
$stmt->close();

if (!$trainer) {
    $error = "Data trainer tidak ditemukan untuk NIK: " . htmlspecialchars($nik_karyawan);
    $nama_trainer = "";
    $foto_profil = null;
    $jabatan = "";
    $divisi = "";
} else {
    $nama_trainer = $trainer['nama_lengkap'];
    $foto_profil = $trainer['foto_profil'];
    $jabatan = $trainer['jabatan'] ?? 'Tidak ada jabatan';
    $divisi = $trainer['divisi'] ?? 'Tidak ada divisi';
}

// Fungsi untuk menghasilkan opsi waktu dalam format 24 jam, kelipatan 30 menit
function generateTimeOptions() {
    $options = [];
    for ($hour = 0; $hour < 24; $hour++) {
        for ($minute = 0; $minute < 60; $minute += 30) {
            $time_24 = sprintf("%02d:%02d", $hour, $minute);
            $options[] = [
                'display' => $time_24,
                'value' => $time_24
            ];
        }
    }
    return $options;
}

$time_options = generateTimeOptions();

// Proses form submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && empty($error)) {
    $judul = clean($_POST['judul']);
    $divisi = trim($_POST['divisi']); // Hanya trim tanpa HTML encoding
    $training_date = $_POST['training_date'];
    $start_time = isset($_POST['start_time']) ? $_POST['start_time'] : '';
    $end_time = isset($_POST['end_time']) ? $_POST['end_time'] : '';
    $peserta_ids = isset($_POST['peserta_ids']) ? $_POST['peserta_ids'] : []; // Array peserta yang dipilih

    // Log input untuk debugging
    file_put_contents('debug.log', print_r([
        'judul' => $judul,
        'divisi' => $divisi,
        'training_date' => $training_date,
        'start_time' => $start_time,
        'end_time' => $end_time,
        'nama_trainer' => $nama_trainer,
        'peserta_ids' => $peserta_ids
    ], true));

    // Validasi input
    if (empty($judul)) {
        $error = "Judul training harus diisi";
    } elseif (empty($divisi)) {
        $error = "Divisi harus dipilih";
    } elseif (empty($training_date)) {
        $error = "Tanggal training harus diisi";
    } elseif (empty($start_time) || empty($end_time)) {
        $error = "Jam mulai dan jam selesai harus diisi";
    } elseif (!preg_match('/^[0-2][0-9]:[0-5][0-9]$/', $start_time) || !preg_match('/^[0-2][0-9]:[0-5][0-9]$/', $end_time)) {
        $error = "Format jam tidak valid";
    } elseif (!isset($_FILES['file']) || $_FILES['file']['error'] != 0) {
        $error = "File training harus diunggah";
    } elseif (empty($nama_trainer)) {
        $error = "Data trainer tidak valid";
    } else {
        // Tambahkan :00 untuk format TIME di database
        $start_time_24 = $start_time . ":00";
        $end_time_24 = $end_time . ":00";

        // Validasi format HH:MM:SS
        if (!preg_match('/^[0-2][0-9]:[0-5][0-9]:[0-5][0-9]$/', $start_time_24) ||
            !preg_match('/^[0-2][0-9]:[0-5][0-9]:[0-5][0-9]$/', $end_time_24)) {
            $error = "Format waktu tidak valid untuk database";
        } else {
            // Validasi tanggal (tidak boleh di masa lalu)
            $today = date('Y-m-d');
            if ($training_date < $today) {
                $error = "Tanggal training tidak boleh di masa lalu";
            }
            // Validasi jam (jam selesai harus setelah jam mulai)
            elseif ($end_time_24 <= $start_time_24) {
                $error = "Jam selesai harus lebih akhir dari jam mulai";
            } else {
                $file = $_FILES['file'];
                $file_name = $file['name'];
                $file_tmp = $file['tmp_name'];
                $file_size = $file['size'];

                // Gunakan pathinfo untuk ekstraksi ekstensi
                if (strpos($file_name, '.') === false) {
                    $error = "Nama file tidak memiliki ekstensi.";
                } else {
                    $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                }

                // Validasi tipe file training
                $allowed_file_types = ['ppt', 'pptx', 'doc', 'docx', 'pdf', 'xls', 'xlsx'];
                if (!in_array($file_ext, $allowed_file_types)) {
                    $error = "Tipe file tidak diizinkan. Hanya file PPT, PPTX, DOC, DOCX, PDF, XLS, XLSX yang diperbolehkan.";
                }
                // Validasi ukuran file (max 20MB)
                elseif ($file_size > 22 * 1024 * 1024) {
                    $error = "Ukuran file training terlalu besar. Maksimal 20MB.";
                } else {
                    // Proses file video (opsional)
                    $video_name = null;
                    $video_ext = null;
                    $video_size = 0;
                    $video_upload_path = null;

                    if (isset($_FILES['video']) && $_FILES['video']['error'] == 0) {
                        $video = $_FILES['video'];
                        $video_name = $video['name'];
                        $video_tmp = $video['tmp_name'];
                        $video_size = $video['size'];

                        $video_ext = strtolower(pathinfo($video_name, PATHINFO_EXTENSION));

                        // Validasi tipe video
                        $allowed_video_types = ['mp4', 'mov', 'avi'];
                        if (!in_array($video_ext, $allowed_video_types)) {
                            $error = "Tipe video tidak diizinkan. Hanya file MP4, MOV, AVI yang diperbolehkan.";
                        }
                        // Validasi ukuran video (max 50MB)
                        elseif ($video_size > 50 * 1024 * 1024) {
                            $error = "Ukuran video terlalu besar. Maksimal 50MB.";
                        }
                    }

                    if (empty($error)) {
                        // Generate nama file unik
                        $new_file_name = time() . '_' . $file_name;
                        $upload_dir = 'Uploads/trainings/';

                        // Buat direktori jika belum ada
                        if (!file_exists($upload_dir)) {
                            mkdir($upload_dir, 0777, true);
                        }

                        $file_upload_path = $upload_dir . $new_file_name;

                        // Proses upload video jika ada
                        $final_video_name = null;
                        $final_video_ext = null;
                        $video_size_value = 0;

                        if ($video_name) {
                            $new_video_name = time() . '_' . $video_name;
                            $video_upload_dir = 'Uploads/videos/';
                            if (!file_exists($video_upload_dir)) {
                                mkdir($video_upload_dir, 0777, true);
                            }
                            $video_upload_path = $video_upload_dir . $new_video_name;
                            $final_video_name = $new_video_name;
                            $final_video_ext = $video_ext;
                            $video_size_value = $video_size;
                        }

                        // Upload file training
                        if (move_uploaded_file($file_tmp, $file_upload_path)) {
                            $upload_success = true;

                            // Upload video jika ada
                            if ($video_name && !move_uploaded_file($video_tmp, $video_upload_path)) {
                                $upload_success = false;
                                $error = "Gagal mengunggah video.";
                                unlink($file_upload_path); // Hapus file training jika video gagal
                            }

                            if ($upload_success) {
                                try {
                                    // Simpan data training ke tabel trainings
                                    $judul_safe = $koneksi->real_escape_string($judul);
                                    $nama_trainer_safe = $koneksi->real_escape_string($nama_trainer);
                                    $divisi_safe = $koneksi->real_escape_string($divisi);
                                    $new_file_name_safe = $koneksi->real_escape_string($new_file_name);
                                    $file_ext_safe = $koneksi->real_escape_string($file_ext);
                                    $file_size_safe = intval($file_size);
                                    $final_video_name_safe = $final_video_name ? "'" . $koneksi->real_escape_string($final_video_name) . "'" : "NULL";
                                    $final_video_ext_safe = $final_video_ext ? "'" . $koneksi->real_escape_string($final_video_ext) . "'" : "NULL";
                                    $video_size_safe = intval($video_size_value);
                                    $training_date_safe = $koneksi->real_escape_string($training_date);
                                    $start_time_24_safe = $koneksi->real_escape_string($start_time_24);
                                    $end_time_24_safe = $koneksi->real_escape_string($end_time_24);

                                    $sql = "INSERT INTO trainings (judul, nama_trainer, divisi, file_name, file_type, file_size, video_name, video_type, video_size, created_at, training_date, start_time, end_time) 
                                            VALUES ('$judul_safe', '$nama_trainer_safe', '$divisi_safe', '$new_file_name_safe', '$file_ext_safe', $file_size_safe, $final_video_name_safe, $final_video_ext_safe, $video_size_safe, NOW(), '$training_date_safe', '$start_time_24_safe', '$end_time_24_safe')";

                                    if ($koneksi->query($sql)) {
                                        $training_id = $koneksi->insert_id; // Ambil ID training yang baru dibuat

                                        // Simpan peserta yang diundang ke tabel training_invitations
                                        if (!empty($peserta_ids)) {
                                            $stmt = $koneksi->prepare("INSERT INTO training_invitations (training_id, peserta_id) VALUES (?, ?)");
                                            foreach ($peserta_ids as $peserta_id) {
                                                $peserta_id = intval($peserta_id); // Pastikan ID valid
                                                $stmt->bind_param("ii", $training_id, $peserta_id);
                                                if (!$stmt->execute()) {
                                                    $error = "Gagal menyimpan undangan peserta: " . $stmt->error;
                                                    // Rollback: Hapus training dan file
                                                    $koneksi->query("DELETE FROM trainings WHERE id = $training_id");
                                                    if (file_exists($file_upload_path)) {
                                                        unlink($file_upload_path);
                                                    }
                                                    if ($video_name && file_exists($video_upload_path)) {
                                                        unlink($video_upload_path);
                                                    }
                                                    break;
                                                }
                                            }
                                            $stmt->close();
                                        }

                                        if (empty($error)) {
                                            $success = "Kegiatan training berhasil ditambahkan!";
                                            // Log sukses untuk debugging
                                            error_log("Training berhasil ditambahkan dengan ID: " . $training_id);
                                        }
                                    } else {
                                        $error = "Gagal menyimpan ke database: " . $koneksi->error;
                                        unlink($file_upload_path);
                                        if ($video_name && file_exists($video_upload_path)) {
                                            unlink($video_upload_path);
                                        }
                                    }
                                } catch (Exception $e) {
                                    $error = "Error: " . $e->getMessage();
                                    if (file_exists($file_upload_path)) {
                                        unlink($file_upload_path);
                                    }
                                    if ($video_name && file_exists($video_upload_path)) {
                                        unlink($video_upload_path);
                                    }
                                }
                            }
                        } else {
                            $error = "Gagal mengunggah file training.";
                        }
                    }
                }
            }
        }
    }
}

// Tambahkan semua ISS ke training_participants (otomatis join)
$iss_query = "SELECT id FROM iss";
$iss_result = $koneksi->query($iss_query);

if ($iss_result && $iss_result->num_rows > 0) {
    $current_time = date('Y-m-d H:i:s');
    
    // Gunakan prepared statement yang lebih aman
    $stmt_iss = $koneksi->prepare("INSERT INTO training_participants (training_id, peserta_id, role, join_time) VALUES (?, ?, ?, ?)");
    
    if ($stmt_iss) {
        $role = 'iss';
        
        while ($iss_user = $iss_result->fetch_assoc()) {
            $iss_id = intval($iss_user['id']);
            $stmt_iss->bind_param("iiss", $training_id, $iss_id, $role, $current_time);
            
            if (!$stmt_iss->execute()) {
                error_log("Gagal menambahkan ISS ID $iss_id ke training ID $training_id: " . $stmt_iss->error);
                // Lanjutkan dengan ISS lainnya, tidak perlu rollback
            }
        }
        
        $stmt_iss->close();
    } else {
        error_log("Gagal menyiapkan statement untuk ISS: " . $koneksi->error);
    }
}

// Set current page untuk highlight menu
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tambah Kegiatan Training - E-Learning</title>
    <link rel="icon" type="image/png" href="image/icon.png">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css"> 
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css"> 
    <style>
        :root {
            --soft-ui-primary-start: #00C4CC; /* Teal */
            --soft-ui-primary-end: #CB0C9F; /* Purple */
            --soft-ui-secondary: #67748E; /* Secondary color */
            --soft-ui-info: #17C1E8; /* Info color */
            --soft-ui-success: #82D616; /* Success color */
            --soft-ui-warning: #FB8C00; /* Warning color */
            --soft-ui-danger: #EA0606; /* Danger color */
            --soft-ui-light: #E9ECEF; /* Light color */
            --soft-ui-dark: #344767; /* Dark color */
            --soft-ui-white: #FFFFFF;
            --soft-ui-gray-100: #F8F9FA;
            --soft-ui-gray-200: #E9ECEF;
            --soft-ui-gray-300: #DEE2E6;
            --soft-ui-gray-400: #CED4DA;
            --soft-ui-gray-500: #ADB5BD;
            --soft-ui-gray-600: #6C757D;
            --soft-ui-gray-700: #495057;
            --soft-ui-gray-800: #343A40;
            --soft-ui-gray-900: #212529;
        }

        html, body {
            height: 100%;
            margin: 0;
            font-family: 'Open Sans', sans-serif;
            background-color: var(--soft-ui-gray-100);
        }
        
        body {
            overflow-x: hidden;
            display: flex;
            flex-direction: column;
        }
        
        .navbar-top {
            position: fixed;
            top: 0;
            width: 100%;
            z-index: 1100;
            background: var(--soft-ui-white);
            border-bottom: 1px solid var(--soft-ui-gray-200);
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.05);
            padding: 0.75rem 1.5rem;
        }
        
        .navbar-top .container-fluid {
            display: flex;
            align-items: center;
            justify-content: space-between;
            flex-wrap: nowrap;
            position: relative;
        }
        
        .navbar-logo {
            height: 40px;
            width: auto;
            margin-right: 15px;
        }
        
        .navbar-title {
            position: absolute;
            left: 50%;
            transform: translateX(-50%);
            font-size: 1.3rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            white-space: nowrap;
            pointer-events: none;
        }

        .navbar-title-mobile {
            display: none;
            font-size: 0.85rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            text-align: center;
            line-height: 1.2;
            flex: 1;
            margin: 0 10px;
        }

        .navbar-title-mobile .line1 {
            display: block;
        }

        .navbar-title-mobile .line2 {
            display: block;
            font-size: 0.8rem;
        }
        
        .navbar-brand {
            font-size: 1.2rem;
            color: var(--soft-ui-dark);
            font-weight: 700;
            display: flex;
            align-items: center;
            z-index: 1;
        }

        .navbar-nav {
            position: relative;
            z-index: 1200;
        }

        .navbar-nav .nav-item .nav-link {
            color: var(--soft-ui-secondary);
            font-weight: 600;
            padding: 0.5rem 1rem;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .nav-item .nav-link:hover {
            color: var(--soft-ui-primary-end);
        }

        .navbar-nav .nav-item .nav-link.active {
            color: var(--soft-ui-primary-end);
        }

        .navbar-nav .dropdown-menu {
            position: absolute;
            top: 100%;
            right: 0;
            left: auto;
            z-index: 1300;
            background-color: var(--soft-ui-white);
            border: 1px solid var(--soft-ui-gray-200);
            border-radius: 0.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            min-width: 160px;
            margin-top: 0.125rem;
        }

        .navbar-nav .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .navbar-nav .dropdown-item:hover {
            background-color: var(--soft-ui-gray-200);
            color: var(--soft-ui-dark);
        }

        .navbar-nav .dropdown-item.active {
            background-color: var(--soft-ui-primary-end);
            color: var(--soft-ui-white);
        }
        
        .sidebar {
            min-height: 100vh;
            width: 250px;
            position: fixed;
            top: 0;
            left: 0;
            z-index: 1000;
            background: var(--soft-ui-white);
            color: var(--soft-ui-dark);
            padding-top: 70px;
            transition: all 0.3s ease-in-out;
            border-right: 1px solid var(--soft-ui-gray-200);
            box-shadow: 4px 0 15px rgba(0, 0, 0, 0.05);
            border-radius: 0 1rem 1rem 0;
        }

        .content {
            margin-left: 250px;
            margin-top: 0;
            padding: 20px;
            padding-top: 90px;
            transition: all 0.3s ease-in-out;
            flex-grow: 1;
        }
        
        .hamburger-mobile {
            display: none;
            background: none;
            border: none;
            color: var(--soft-ui-dark);
            font-size: 1.5rem;
            cursor: pointer;
            margin-right: 15px;
            transition: all 0.3s ease;
            z-index: 1;
        }
        
        .hamburger-mobile:focus {
            outline: none;
        }
        
        .sidebar-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(0, 0, 0, 0.4);
            z-index: 999;
        }
        
        .profile-section {
            text-align: center;
            padding: 15px 20px;
            border-bottom: 1px solid var(--soft-ui-gray-200);
            background: var(--soft-ui-white);
            border-radius: 0.75rem;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .profile-photo {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid var(--soft-ui-gray-300);
            margin: 0 auto 10px;
            display: block;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: var(--soft-ui-gray-400);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            border: 2px solid var(--soft-ui-gray-300);
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.08);
        }
        
        .profile-photo-placeholder i {
            font-size: 2rem;
            color: var(--soft-ui-white);
        }
        
        .profile-name {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--soft-ui-dark);
            margin-bottom: 5px;
            word-wrap: break-word;
        }
        
        .profile-info {
            font-size: 0.9rem;
            font-weight: 400;
            color: var(--soft-ui-secondary);
            margin-bottom: 3px;
        }
        
        .sidebar .nav-link {
            color: var(--soft-ui-secondary);
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            margin: 0.25rem 0.75rem;
            transition: all 0.2s ease-in-out;
            font-weight: 600;
            display: flex;
            align-items: center;
        }
        
        .sidebar .nav-link:hover {
            background: linear-gradient(90deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-dark);
        }
        
        .sidebar .nav-link.active {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white) !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .sidebar .nav-link.active i {
            color: var(--soft-ui-white);
        }
        
        .sidebar .nav-link i {
            font-size: 1.2rem;
            margin-right: 10px;
            color: var(--soft-ui-secondary);
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-menu {
            background-color: var(--soft-ui-white);
            border: none;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            border-radius: 0.5rem;
            padding: 0.5rem 0;
            margin-top: 0.25rem;
        }

        .sidebar .nav-item.dropdown .dropdown-item {
            color: var(--soft-ui-secondary);
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.2s ease-in-out;
        }

        .sidebar .nav-item.dropdown .dropdown-item:hover {
            background: linear-gradient(90deg, rgba(0, 196, 204, 0.1), rgba(203, 12, 159, 0.1));
            color: var(--soft-ui-dark);
        }

        .sidebar .nav-item.dropdown .dropdown-item.active {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
        }

        .sidebar .nav-item:not(:last-child) {
            border-bottom: none;
        }

        .page-title {
            font-size: 2rem;
            font-weight: 700;
            color: var(--soft-ui-dark);
            margin-bottom: 30px;
            text-align: center;
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .card {
            background-color: var(--soft-ui-white);
            border-radius: 0.75rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            transition: all 0.2s ease-in-out;
            border: none;
        }

        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.08);
        }

        .form-card {
            height: 100%;
        }

        .form-header {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
            font-weight: 700;
            font-size: 1.25rem;
            text-align: center;
            padding: 1.5rem;
            border-top-left-radius: 0.75rem;
            border-top-right-radius: 0.75rem;
        }

        .form-body {
            padding: 2rem;
        }

        .form-label {
            font-weight: 600;
            color: var(--soft-ui-dark);
            margin-bottom: 0.5rem;
        }

        .form-control, .form-select {
            border: 1px solid var(--soft-ui-gray-300);
            border-radius: 0.5rem;
            padding: 0.75rem 1rem;
            font-size: 1rem;
            transition: all 0.2s ease-in-out;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--soft-ui-primary-start);
            box-shadow: 0 0 0 0.2rem rgba(0, 196, 204, 0.25);
        }

        .form-text {
            color: var(--soft-ui-secondary);
            font-size: 0.875rem;
            margin-top: 0.25rem;
        }

        .btn {
            border-radius: 0.5rem;
            padding: 0.75rem 1.5rem;
            font-weight: 600;
            transition: all 0.2s ease-in-out;
            border: none;
        }

        .btn-primary {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
        }

        .btn-primary:hover {
            background: linear-gradient(195deg, #009ea6, #a80a85);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 196, 204, 0.3);
            color: var(--soft-ui-white);
        }

        .btn-secondary {
            background: var(--soft-ui-secondary);
            color: var(--soft-ui-white);
        }

        .btn-secondary:hover {
            background: #5a6c7d;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(103, 116, 142, 0.3);
            color: var(--soft-ui-white);
        }

        .alert {
            border-radius: 0.75rem;
            border: none;
            padding: 1rem 1.5rem;
            font-weight: 500;
        }

        .alert-danger {
            background: rgba(0, 196, 204, 0.1);
            color: var(--soft-ui-primary-start);
        }

        .alert-success {
            background: rgba(203, 12, 159, 0.1);
            color: var(--soft-ui-primary-end);
        }

        .alert-link {
            color: var(--soft-ui-primary-end);
            font-weight: 600;
            text-decoration: none;
        }

        .alert-link:hover {
            color: #a80a85;
            text-decoration: underline;
        }

        .autocomplete-container {
            position: relative;
        }
        
        .autocomplete-suggestions {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            z-index: 1000;
            background: var(--soft-ui-white);
            border: 1px solid var(--soft-ui-gray-200);
            border-radius: 0.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            max-height: 200px;
            overflow-y: auto;
            display: none;
        }
        
        .autocomplete-suggestions.show {
            display: block;
        }
        
        .autocomplete-suggestion {
            padding: 0.75rem 1rem;
            cursor: pointer;
            transition: background 0.2s ease;
            border-bottom: 1px solid var(--soft-ui-gray-200);
        }

        .autocomplete-suggestion:last-child {
            border-bottom: none;
        }
        
        .autocomplete-suggestion:hover {
            background: var(--soft-ui-gray-100);
        }
        
        .autocomplete-suggestion.highlighted {
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
        }
        
        .tags-container {
            display: flex;
            flex-wrap: wrap;
            gap: 0.5rem;
            min-height: 48px;
            border: 1px solid var(--soft-ui-gray-300);
            border-radius: 0.5rem;
            padding: 0.5rem;
            background: var(--soft-ui-white);
            transition: all 0.2s ease-in-out;
        }

        .tags-container:focus-within {
            border-color: var(--soft-ui-primary-start);
            box-shadow: 0 0 0 0.2rem rgba(0, 196, 204, 0.25);
        }
        
        .tag {
            display: inline-flex;
            align-items: center;
            background: linear-gradient(195deg, var(--soft-ui-primary-start), var(--soft-ui-primary-end));
            color: var(--soft-ui-white);
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
            font-size: 0.875rem;
            font-weight: 500;
        }
        
        .tag-remove {
            margin-left: 0.5rem;
            cursor: pointer;
            font-weight: 700;
            transition: opacity 0.2s ease;
        }

        .tag-remove:hover {
            opacity: 0.7;
        }
        
        .tags-input {
            border: none;
            outline: none;
            flex-grow: 1;
            padding: 0.25rem;
            font-size: 1rem;
            min-width: 120px;
        }
        
        .tags-input:focus {
            border: none;
            box-shadow: none;
        }
        
        footer {
            background: var(--soft-ui-white);
            color: var(--soft-ui-secondary);
            border-top: 1px solid var(--soft-ui-gray-200);
            padding: 1rem 0;
            text-align: center;
            flex-shrink: 0;
            margin-top: 2rem;
        }
        
        .animate-fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        @media (max-width: 768px) {
            .sidebar {
                margin-left: -250px;
                border-radius: 0;
                padding-top: 90px;
            }
            .sidebar-active {
                margin-left: 0;
            }
            .content {
                margin-left: 0;
                padding-top: 100px;
            }
            
            .navbar-title {
                display: none;
            }
            .navbar-logo {
                display: none;
            }
            
            .navbar-title-mobile {
                display: block;
            }
            
            .hamburger-mobile {
                display: block;
            }
            .navbar-search {
                display: none;
            }
            .navbar-nav {
                position: static;
            }
            .navbar-nav .dropdown-menu {
                position: absolute;
                right: 0;
                left: auto;
                z-index: 1300;
            }
            .page-title {
                font-size: 1.5rem;
                margin-bottom: 20px;
            }
            
            .form-body {
                padding: 1rem;
            }

            .form-control, .form-select {
                font-size: 0.9rem;
                padding: 0.5rem 0.75rem;
            }

            .btn {
                padding: 0.5rem 1rem;
                font-size: 0.9rem;
            }

            .autocomplete-suggestions {
                max-height: 150px;
            }

            .tags-container {
                min-height: 42px;
            }

            .tag {
                font-size: 0.8rem;
                padding: 0.2rem 0.4rem;
            }
        }

        @media (max-width: 576px) {
            .navbar-top {
                padding: 0.5rem 1rem;
                min-height: 70px;
            }
            .navbar-title-mobile {
                font-size: 0.8rem;
            }
            .navbar-title-mobile .line2 {
                font-size: 0.75rem;
            }
            .page-title {
                font-size: 1.25rem;
            }

            .form-body {
                padding: 0.75rem;
            }

            .form-control, .form-select {
                font-size: 0.8rem;
                padding: 0.5rem;
            }

            .btn {
                padding: 0.4rem 0.8rem;
                font-size: 0.8rem;
            }

            .form-text {
                font-size: 0.75rem;
            }

            .tag {
                font-size: 0.75rem;
                padding: 0.2rem 0.4rem;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-top">
        <div class="container-fluid">
            <button class="hamburger-mobile d-md-none" type="button" id="sidebarToggle">
                <i class="bi bi-list"></i>
            </button>
            <a class="navbar-brand d-none d-md-flex" href="#">
                <img src="image/logopim.png" alt="Logo PIM" class="navbar-logo">
            </a>
            <div class="navbar-title d-none d-md-block">
                E-Learning System Pondok Indah Mall
            </div>
            <div class="navbar-title-mobile d-md-none">
                <span class="line1">E-Learning System</span>
                <span class="line2">Pondok Indah Mall</span>
            </div>
            <ul class="navbar-nav ms-auto mb-2 mb-lg-0">
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle text-dark" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-person-circle"></i>
                    </a>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><a class="dropdown-item <?php echo ($current_page == 'profile.php') ? 'active' : ''; ?>" href="profile.php">Profile</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="logout.php">Logout</a></li>
                    </ul>
                </li>
            </ul>
        </div>
    </nav>

    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <div class="sidebar border-end" id="sidebar">
        <div class="profile-section">
            <?php if (!empty($foto_profil)): ?>
                <img src="Uploads/profiles/<?php echo htmlspecialchars($foto_profil); ?>" alt="Foto Profil" class="profile-photo">
            <?php else: ?>
                <div class="profile-photo-placeholder">
                    <i class="bi bi-person-circle"></i>
                </div>
            <?php endif; ?>
            <div class="profile-name"><?php echo htmlspecialchars($nama_trainer); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($jabatan); ?></div>
            <div class="profile-info"><?php echo htmlspecialchars($divisi); ?></div>
        </div>
        <div class="p-3">
            <h5 class="mb-4 text-dark">Menu</h5>
            <ul class="nav flex-column">
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'dashboard.php') ? 'active' : ''; ?>" href="dashboard.php">
                        <i class="bi bi-house-door-fill"></i> Dashboard
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo ($current_page == 'materi.php') ? 'active' : ''; ?>" href="materi.php">
                        <i class="bi bi-book"></i> Materi
                    </a>
                </li>
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_posttest.php' || $current_page == 'list_posttest.php') ? 'active' : ''; ?>" href="#" id="postTestDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-clipboard-check"></i> Post Test
                    </a>
                    <ul class="dropdown-menu" aria-labelledby="postTestDropdown">
                        <li><a class="dropdown-item <?php echo ($current_page == 'add_posttest.php') ? 'active' : ''; ?>" href="add_posttest.php">Tambah Post Test</a></li>
                        <li><a class="dropdown-item <?php echo ($current_page == 'list_posttest.php') ? 'active' : ''; ?>" href="list_posttest.php">List Post Test</a></li>
                    </ul>
                </li>
                <li class="nav-item dropdown">
                    <a class="nav-link dropdown-toggle <?php echo ($current_page == 'add_training.php' || $current_page == 'list_training.php') ? 'active' : ''; ?>" 
                       href="#" id="trainingDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <i class="bi bi-journal-text"></i> Training
                    </a>
                    <ul class="dropdown-menu" aria-labelledby="trainingDropdown">
                        <li><a class="dropdown-item <?php echo ($current_page == 'add_training.php') ? 'active' : ''; ?>" href="add_training.php">Tambah Kegiatan Training</a></li>
                        <li><a class="dropdown-item <?php echo ($current_page == 'list_training.php') ? 'active' : ''; ?>" href="list_training.php">List Training</a></li>
                    </ul>
                </li>
                
                <?php if ($_SESSION['role'] == 'trainer'): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'report.php') ? 'active' : ''; ?>" href="report.php">
                            <i class="bi bi-file-earmark-text"></i> Report
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo ($current_page == 'data_evaluasi.php') ? 'active' : ''; ?>" href="data_evaluasi.php">
                            <i class="bi bi-clipboard-data"></i> Data Evaluasi
                        </a>
                    </li>
                <?php endif; ?>
            </ul>
        </div>
    </div>

    <div class="content">
        <div class="container-fluid">
            <h2 class="page-title animate-fade-in">Tambah Kegiatan Training</h2>
            <div class="row">
                <div class="col-md-10 offset-md-1">
                    <div class="card form-card animate-fade-in">
                        <div class="form-header">
                            <i class="bi bi-journal-plus me-2"></i>
                            Form Kegiatan Training
                        </div>
                        <div class="form-body">
                            <?php if (!empty($error)): ?>
                                <div class="alert alert-danger">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <?= htmlspecialchars($error) ?>
                                </div>
                            <?php endif; ?>
                            <?php if (!empty($success)): ?>
                                <div class="alert alert-success">
                                    <i class="bi bi-check-circle me-2"></i>
                                    <?= htmlspecialchars($success) ?> 
                                    <a href="dashboard.php" class="alert-link">Kembali ke Dashboard</a>
                                </div>
                            <?php endif; ?>
                            
                            <form method="POST" action="" enctype="multipart/form-data">
                                <div class="mb-3">
                                    <label for="judul" class="form-label">
                                        <i class="bi bi-card-text me-2"></i>Judul Training <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" class="form-control" id="judul" name="judul" required>
                                </div>
                                <div class="mb-3">
                                    <label for="nama_trainer" class="form-label">
                                        <i class="bi bi-person-badge me-2"></i>Nama Trainer <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" class="form-control" id="nama_trainer" name="nama_trainer" value="<?php echo htmlspecialchars($nama_trainer); ?>" readonly>
                                </div>
                                <div class="mb-3">
                                    <label for="divisi" class="form-label">
                                        <i class="bi bi-building me-2"></i>Divisi <span class="text-danger">*</span>
                                    </label>
                                    <select class="form-select" id="divisi" name="divisi" required>
                                        <option value="" disabled selected>Pilih Divisi</option>
                                        <option value="Customer Service">Customer Service</option>
                                        <option value="Engineering">Engineering</option>
                                        <option value="Engineering-BM">Engineering-BM</option>
                                        <option value="Engineering-Fit Out">Engineering-Fit Out</option>
                                        <option value="Engineering Landscape">Engineering Landscape</option>
                                        <option value="Engineering-ME">Engineering-ME</option>
                                        <option value="Engineering-Waterpark">Engineering-Waterpark</option>
                                        <option value="Finance">Finance</option>
                                        <option value="HSE & CCTV">HSE & CCTV</option>
                                        <option value="Human Resource & General Affair">Human Resource & General Affair</option>
                                        <option value="IT">IT</option>
                                        <option value="Legal">Legal</option>
                                        <option value="Marketing">Marketing</option>
                                        <option value="Parkir">Parkir</option>
                                        <option value="Promosi">Promosi</option>
                                        <option value="Security">Security</option>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label for="peserta_input" class="form-label">
                                        <i class="bi bi-people me-2"></i>Pilih Peserta <span class="text-danger">*</span>
                                    </label>
                                    <div class="autocomplete-container">
                                        <div class="tags-container" id="tags-container">
                                            <input type="text" class="tags-input" id="peserta_input" placeholder="Cari peserta..." required>
                                        </div>
                                        <div class="autocomplete-suggestions" id="peserta_suggestions"></div>
                                    </div>
                                    <div class="form-text">
                                        Ketik nama peserta untuk mencari. Klik nama atau gunakan panah dan Enter untuk memilih. Minimal pilih 1 peserta.
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <label for="file" class="form-label">
                                        <i class="bi bi-file-earmark me-2"></i>File Training <span class="text-danger">*</span>
                                    </label>
                                    <input type="file" class="form-control" id="file" name="file" required>
                                    <div class="form-text">
                                        Format file yang diizinkan: PPT, PPTX, DOC, DOCX, PDF, XLS, XLSX. Ukuran maksimal: 20MB.
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <label for="video" class="form-label">
                                        <i class="bi bi-camera-video me-2"></i>Video Training (Opsional)
                                    </label>
                                    <input type="file" class="form-control" id="video" name="video">
                                    <div class="form-text">
                                        Format file yang diizinkan: MP4, MOV, AVI. Ukuran maksimal: 50MB.
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">
                                        <i class="bi bi-calendar-event me-2"></i>Jadwal Training <span class="text-danger">*</span>
                                    </label>
                                    <div class="row">
                                        <div class="col-md-4">
                                            <label for="training_date" class="form-label">Tanggal</label>
                                            <input type="date" class="form-control" id="training_date" name="training_date" required>
                                        </div>
                                        <div class="col-md-4">
                                            <label for="start_time" class="form-label">Jam Mulai</label>
                                            <select class="form-select" id="start_time" name="start_time" required>
                                                <option value="" disabled selected>Pilih Jam</option>
                                                <?php foreach ($time_options as $option): ?>
                                                    <option value="<?php echo $option['value']; ?>"><?php echo $option['display']; ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        <div class="col-md-4">
                                            <label for="end_time" class="form-label">Jam Selesai</label>
                                            <select class="form-select" id="end_time" name="end_time" required>
                                                <option value="" disabled selected>Pilih Jam</option>
                                                <?php foreach ($time_options as $option): ?>
                                                    <option value="<?php echo $option['value']; ?>"><?php echo $option['display']; ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-save me-2"></i> Simpan Training
                                    </button>
                                    <a href="dashboard.php" class="btn btn-secondary">
                                        <i class="bi bi-x-circle me-2"></i> Batal
                                    </a>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <footer class="text-center">
        <div class="container">
            <p class="mb-0">© <?php echo date('Y'); ?> E-Learning System by PIM</p>
        </div>
    </footer>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script> 
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script> 
    <script>
        if (typeof bootstrap === 'undefined') {
            console.error('Bootstrap JavaScript tidak dimuat. Periksa koneksi jaringan atau CDN.');
        }

        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebarToggle');
            const sidebar = document.getElementById('sidebar');
            const sidebarOverlay = document.getElementById('sidebarOverlay');
            const trainingDropdownToggle = document.getElementById('trainingDropdown');
            const postTestDropdownToggle = document.getElementById('postTestDropdown');

            let isDropdownClick = false;

            function toggleSidebar() {
                sidebar.classList.toggle('sidebar-active');
                sidebarOverlay.style.display = sidebar.classList.contains('sidebar-active') ? 'block' : 'none';

                if (sidebar.classList.contains('sidebar-active')) {
                    document.body.style.overflow = 'hidden';
                } else {
                    document.body.style.overflow = '';
                }
            }

            sidebarToggle.addEventListener('click', function(e) {
                e.stopPropagation();
                toggleSidebar();
            });

            sidebarOverlay.addEventListener('click', function() {
                if (!isDropdownClick) {
                    toggleSidebar();
                }
                isDropdownClick = false;
            });

            if (trainingDropdownToggle) {
                trainingDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }

            if (postTestDropdownToggle) {
                postTestDropdownToggle.addEventListener('click', function(e) {
                    if (window.innerWidth <= 768) {
                        isDropdownClick = true;
                    }
                });
            }

            document.addEventListener('click', function(e) {
                if (window.innerWidth <= 768) {
                    let clickedInsideAnyDropdown = false;
                    if (trainingDropdownToggle && trainingDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (postTestDropdownToggle && postTestDropdownToggle.contains(e.target)) {
                        clickedInsideAnyDropdown = true;
                    }
                    if (!clickedInsideAnyDropdown) {
                        isDropdownClick = false;
                    }
                }
            });

            const regularNavLinks = document.querySelectorAll('.sidebar .nav-link:not(.dropdown-toggle)');
            regularNavLinks.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            const dropdownItems = document.querySelectorAll('.sidebar .dropdown-menu .dropdown-item');
            dropdownItems.forEach(item => {
                item.addEventListener('click', function() {
                    if (window.innerWidth <= 768 && sidebar.classList.contains('sidebar-active')) {
                        toggleSidebar();
                    }
                });
            });

            window.addEventListener('resize', function() {
                if (window.innerWidth > 768) {
                    sidebar.classList.remove('sidebar-active');
                    sidebarOverlay.style.display = 'none';
                    document.body.style.overflow = '';
                    isDropdownClick = false;
                }
            });
        });

        document.getElementById('training_date').addEventListener('change', function() {
            const today = new Date().toISOString().split('T')[0];
            if (this.value < today) {
                this.setCustomValidity('Tanggal training tidak boleh di masa lalu');
            } else {
                this.setCustomValidity('');
            }
        });

        document.getElementById('end_time').addEventListener('change', function() {
            const startTime = document.getElementById('start_time').value;
            const endTime = this.value;
            if (startTime && endTime) {
                const startParts = startTime.split(':');
                const endParts = endTime.split(':');
                const startMinutes = parseInt(startParts[0]) * 60 + parseInt(startParts[1]);
                const endMinutes = parseInt(endParts[0]) * 60 + parseInt(endParts[1]);
                if (endMinutes <= startMinutes) {
                    this.setCustomValidity('Jam selesai harus lebih akhir dari jam mulai');
                } else {
                    this.setCustomValidity('');
                }
            }
        });

        document.querySelector('form').addEventListener('submit', function(e) {
            const selectedPeserta = document.querySelectorAll('input[name="peserta_ids[]"]');
            if (selectedPeserta.length === 0) {
                e.preventDefault();
                alert('Minimal pilih 1 peserta untuk training ini.');
                return false;
            }
        });

        $(document).ready(function() {
            const pesertaInput = $('#peserta_input');
            const suggestionsContainer = $('#peserta_suggestions');
            const tagsContainer = $('#tags-container');
            let selectedPeserta = [];
            let highlightedIndex = -1;

            function addTag(id, name, nik) {
                if (!selectedPeserta.some(p => p.id === id)) {
                    selectedPeserta.push({ id, name, nik });
                    const tagHtml = `
                        <span class="tag" data-id="${id}">
                            ${name} (${nik})
                            <span class="tag-remove" data-id="${id}">×</span>
                            <input type="hidden" name="peserta_ids[]" value="${id}">
                        </span>
                    `;
                    tagsContainer.find('.tags-input').before(tagHtml);
                    pesertaInput.val('');
                    suggestionsContainer.removeClass('show').empty();
                    highlightedIndex = -1;
                    
                    pesertaInput.removeAttr('required');
                }
            }

            tagsContainer.on('click', '.tag-remove', function() {
                const id = $(this).data('id');
                selectedPeserta = selectedPeserta.filter(p => p.id !== id);
                $(this).parent().remove();
                
                if (selectedPeserta.length === 0) {
                    pesertaInput.attr('required', 'required');
                }
            });

            function updateHighlight() {
                suggestionsContainer.find('.autocomplete-suggestion').removeClass('highlighted');
                if (highlightedIndex >= 0) {
                    suggestionsContainer.find('.autocomplete-suggestion').eq(highlightedIndex).addClass('highlighted');
                }
            }

            pesertaInput.on('input', function() {
                const query = $(this).val().trim();
                highlightedIndex = -1;
                if (query.length === 0) {
                    suggestionsContainer.removeClass('show').empty();
                    return;
                }

                $.ajax({
                    url: 'get_peserta_suggestions.php',
                    method: 'GET',
                    data: { query: query },
                    dataType: 'json',
                    success: function(data) {
                        suggestionsContainer.empty();
                        if (data.length > 0) {
                            data.forEach(peserta => {
                                if (!selectedPeserta.some(p => p.id === peserta.id)) {
                                    const suggestionHtml = `
                                        <div class="autocomplete-suggestion" data-id="${peserta.id}" data-name="${peserta.nama_lengkap}" data-nik="${peserta.nik_karyawan}">
                                            ${peserta.nama_lengkap} (${peserta.nik_karyawan})
                                        </div>
                                    `;
                                    suggestionsContainer.append(suggestionHtml);
                                }
                            });
                            suggestionsContainer.addClass('show');
                        } else {
                            suggestionsContainer.removeClass('show');
                        }
                    },
                    error: function() {
                        suggestionsContainer.removeClass('show').empty();
                    }
                });
            });

            suggestionsContainer.on('click', '.autocomplete-suggestion', function() {
                const id = $(this).data('id');
                const name = $(this).data('name');
                const nik = $(this).data('nik');
                addTag(id, name, nik);
            });

            pesertaInput.on('keydown', function(e) {
                const suggestions = suggestionsContainer.find('.autocomplete-suggestion');
                const suggestionCount = suggestions.length;

                if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    if (highlightedIndex < suggestionCount - 1) {
                        highlightedIndex++;
                        updateHighlight();
                    }
                } else if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    if (highlightedIndex > 0) {
                        highlightedIndex--;
                        updateHighlight();
                    }
                } else if (e.key === 'Enter' && highlightedIndex >= 0) {
                    e.preventDefault();
                    const selectedSuggestion = suggestions.eq(highlightedIndex);
                    const id = selectedSuggestion.data('id');
                    const name = selectedSuggestion.data('name');
                    const nik = selectedSuggestion.data('nik');
                    addTag(id, name, nik);
                }
            });

            $(document).on('click', function(e) {
                if (!$(e.target).closest('.autocomplete-container').length) {
                    suggestionsContainer.removeClass('show').empty();
                    highlightedIndex = -1;
                }
            });
        });
    </script>
</body>
</html>